/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VirtualMachineMXBean.java 5764 2009-12-14 20:33:59Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.api;

import java.util.Date;

import javax.management.InstanceNotFoundException;

/**
 * Management interface of a virtual machine. The <tt>ObjectName</tt> for
 * identifying a VirtualMachineMXBean follows the following template:
 * <p>
 * <code>
 *    org.ow2.jasmine.vmm.api:type=VirtualMachine,name=&lt;pathname&gt;,uuid=&lt;uuid&gt;
 * </code>
 * <p>
 * The notifications that can be emitted by a HostMXBean are as follows
 * <ul>
 * <li>NotificationTypes.VM_MIGRATION when the VM has migrated to a new host
 * <li>NotificationTypes.VM_STATE_CHANGE when the power state of the VM has
 * changed
 * </ul>
 */
public interface VirtualMachineMXBean extends ManagedResourceMXBean {
    /**
     * Different states of a virtual machine.
     */
    public enum PowerState {
        /**
         * the virtual machine is halted
         */
        HALTED,
        /**
         * the virtual machine is paused
         */
        PAUSED,
        /**
         * The virtual machine is running
         */
        RUNNING,
        /**
         * the virtual machine is suspended
         */
        SUSPENDED,
        /**
         * the state of the virtual machine is unknown
         */
        UNKNOWN
    };

    /**
     * Returns the Uuid of the virtual machine
     * 
     * @return
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public String getUuid() throws InstanceNotFoundException;

    /**
     * Returns the host this VM resides on
     * 
     * @return
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public HostMXBean getHostMBean() throws InstanceNotFoundException;

    /**
     * Returns the name of the virtual machine
     * 
     * @return
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public String getNameLabel() throws InstanceNotFoundException;

    /**
     * Returns the MAC addresses of all network interfaces associated with this
     * VM
     * 
     * @return MAC addresses in xx:xx:xx:xx:xx:xx format
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public String getMacAddress() throws InstanceNotFoundException;

    /**
     * Returns the IP addresses of the guest operating system running within the
     * VM.
     * 
     * @return IP addresses or null if the manager is unable to determine the IP
     *         addresses
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public String getGuestIpAddress() throws InstanceNotFoundException;

    /**
     * Returns the load per virtual CPU of the VM
     * 
     * @return
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public float[] getLoadPerVCPU() throws InstanceNotFoundException;

    /**
     * Returns the CPU load of the VM
     * 
     * @return
     * @throws InstanceNotFoundException
     */
    public float getCPULoad() throws InstanceNotFoundException;

    /**
     * Returns the start time of the virtual machine
     * 
     * @return
     * @throws InstanceNotFoundException
     */
    public Date getStartTime() throws InstanceNotFoundException;

    /**
     * Returns the memory size in MBytes of the virtual machine
     * 
     * @return
     * @throws InstanceNotFoundException
     */
    public long getMemorySizeMB() throws InstanceNotFoundException;

    /**
     * Attempts to change the memory sie of the virtual machine
     * 
     * @param size target size
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public void setMemorySizeMB(long size) throws InstanceNotFoundException;

    /**
     * Returns the amount of memory (in MBytes) used by the virtual machine
     * 
     * @return
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public long getMemoryUsedMB() throws InstanceNotFoundException;

    /**
     * Return the current resource usage (CPU, memory, network, disk I/O) for
     * this VM
     * 
     * @return a ResourceUsage object representing the resource usage of this VM
     */
    public ResourceUsage getResourceUsage();

    /**
     * Returns the number of virtual CPUs assigned to the virtual machine
     * 
     * @return
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public int getNumVCPUs() throws InstanceNotFoundException;

    /**
     * Changes the number of virtual CPUs assigned to the virtual machine
     * 
     * @param numVCPUs
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public void setNumVCPUs(int numVCPUs) throws InstanceNotFoundException;

    /**
     * Returns the scheduling cap assigned to the VM
     * 
     * @return current scheduling cap
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public int getSchedulingCap() throws InstanceNotFoundException;

    /**
     * Sets the scheduling cap assigned to the VM
     * 
     * @param schedulingCap new cap
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public void setSchedulingCap(int schedulingCap) throws InstanceNotFoundException;

    /**
     * Returns the scheduling weight assigned to the VM
     * 
     * @return
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public int getSchedulingWeight() throws InstanceNotFoundException;

    /**
     * Sets the scheduling cap assigned to the VM
     * 
     * @param schedulingWeight new weight
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public void setSchedulingWeight(int schedulingWeight) throws InstanceNotFoundException;

    /**
     * Returns the power state of the VM
     * 
     * @return current power state
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public PowerState getState() throws InstanceNotFoundException;

    /**
     * Suspend the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws BadVMPowerStateException thrown if the VM is not in the
     *         appropriate state
     */
    public void suspend() throws InstanceNotFoundException, BadVMPowerStateException;

    /**
     * Resume the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws BadVMPowerStateException thrown if the VM is not in the
     *         appropriate state
     */
    public void resume() throws InstanceNotFoundException, BadVMPowerStateException;

    /**
     * Shutdown the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws BadVMPowerStateException thrown if the VM is not in the
     *         appropriate state
     */
    public void shutdown() throws InstanceNotFoundException, BadVMPowerStateException;

    /**
     * Start the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws BadVMPowerStateException thrown if the VM is not in the
     *         appropriate state
     */
    public void start() throws InstanceNotFoundException, BadVMPowerStateException;

    /**
     * Pause the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws BadVMPowerStateException thrown if the VM is not in the
     *         appropriate state
     */
    public void pause() throws InstanceNotFoundException, BadVMPowerStateException;

    /**
     * Unpause the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws BadVMPowerStateException thrown if the VM is not in the
     *         appropriate state
     */
    public void unpause() throws InstanceNotFoundException, BadVMPowerStateException;

    /**
     * Reboot the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws BadVMPowerStateException thrown if the VM is not in the
     *         appropriate state
     */
    public void reboot() throws InstanceNotFoundException, BadVMPowerStateException;

    /**
     * Destroy the VM
     * 
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     */
    public void destroy() throws InstanceNotFoundException;

    /**
     * Returns true if the VM can be migrated to the given host using live
     * migration
     * 
     * @param targetHost host on which the VM migration is tested
     */
    public boolean canLiveMigrateToHost(HostMXBean targetHost);

    /**
     * Migrate the VM to a target host. Note that the target host must belong to
     * the same server pool as the VM's current host.
     * 
     * @param targetHost target host of the migration
     * @param live if true a live migration will be attempted
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws IllegalOperationException thrown if the migration cannot be
     *         performed
     */
    public void migrate(HostMXBean targetHost, boolean live) throws InstanceNotFoundException, IllegalOperationException,
        VMMException;

    /**
     * Returns user data associated with the specified key
     * 
     * @param key of the user data to return
     * @return user data for the specified key
     */
    public String getUserData(String key) throws InstanceNotFoundException;

    /**
     * Clones the VM
     * 
     * @param vmName name to be given to the new VM
     * @param custSpec optional customization parameters
     * @param sync if true, the operation is synchronous
     * @return the management interface of the cloned VM if <tt>sync</tt> is
     *         true, null otherwise
     * @throws InstanceNotFoundException thrown if the specified MXBean does not
     *         exist
     * @throws InsufficientResourcesException raised if not enough resources are
     *         available to clone the VM
     */
    public VirtualMachineMXBean cloneVM(String vmName, VMCustomizationSpec custSpec, boolean sync)
        throws InstanceNotFoundException, InsufficientResourcesException, VMMException;

    /**
     * Makes a VM image template from this VM
     * 
     * @param vmImageID name of the VM image to create
     * @throws InsufficientResourcesException thrown if not enough resources are
     *         available
     * @throws IllegalOperationException
     * @throws RuntimeVirtException
     */
    public void makeTemplate(String vmImageID, String name, String description) throws InsufficientResourcesException,
        IllegalOperationException, VMMException;

}
