/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DomainMXBean.java 6167 2010-03-23 16:07:16Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.api;

import java.util.List;
import java.util.Map;

/**
 * Management interface for a domain. A domain is a management unit consisting
 * of subdomains and server pools. Domains are organized in a hierarchical
 * fashion. Domains are named using a directory-like naming scheme. Each domain
 * has a local name within its parent domain (e.g. "ParisDatacenter") and an
 * absolute name (e.g. "/root/paris/ParisDatacenter").
 * <p>
 * The <tt>ObjectName</tt> for identifying a DomainMXBean follows the following
 * template: <blockquote>
 * org.ow2.jasmine.vmm.api:type=Domain,name=&lt;pathname&gt; </blockquote>
 */
public interface DomainMXBean extends ManagedResourceMXBean {
    /**
     * Return the symbolic name of the domain
     * 
     * @return
     */
    String getName();

    /**
     * Returns an attribute of the domain
     * 
     * @return the name of the attribute requested
     */
    String getAttribute(String key);

    /**
     * Provisions a new virtual machine. The host, server pool (and possibly)
     * subdomain on which the VM will be created will depend on the provisioning
     * policy active for this domain (and its subdomains). Upon successful
     * creation a notification of type NotificationTypes.VM_NEW will be emitted
     * by the HostMXBean on which the VM has been created.
     * 
     * @param vmSpec initial configuration of the virtual machine
     * @param constraints contraints on the placement of the VM
     * @param sync if true the call will return once the VM has been created
     * @return the management interface of the created VM if <tt>sync</tt> is
     *         true, null otherwise
     * @throws InsufficientResourcesException raised if not enough resources are
     *         available to create the VM
     * @throws InvalidVMConfigException raised if the VM configuration is
     *         invalid
     * @throws VMMException raised if any type of runtime fault occurs that is
     *         not covered by the previous exceptions
     */
    VirtualMachineMXBean provisionVM(VMConfigSpec vmSpec, Map<String, String> constraints, boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException;

    /**
     * Returns a list of the server pools belonging to this domain
     * 
     * @return
     */
    List<ServerPoolMXBean> getServerPools();

    /**
     * Returns a list of the subdomains of this domain
     * 
     * @return
     */
    List<DomainMXBean> getSubDomains();

}
