/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ResourceUsage.java 6093 2010-02-23 13:15:50Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.api;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.management.openmbean.CompositeData;

/**
 * A snapshot of performance metrics for a virtual machine
 */
public class ResourceUsage implements Serializable {
    private static final long serialVersionUID = -4750072156087484043L;

    /**
     * Disk I/O performance metrics of a device
     */
    public static class DiskStats implements Serializable {
        private static final long serialVersionUID = -3692224237657640220L;

        private String deviceName;

        private long diskReadKBytePerSec;

        private long diskWrittenKBytesPerSec;

        /**
         * Construct a new DiskStats instance
         */
        public DiskStats() {
        }

        /**
         * Constructs a new DiskStats instance
         * 
         * @param deviceName name of the device
         * @param diskReadKBytePerSec number of kilobytes read per second
         * @param diskWrittenKBytesPerSec number of kilobytes written per second
         */
        public DiskStats(final String deviceName, final long diskReadKBytePerSec, final long diskWrittenKBytesPerSec) {
            this.deviceName = deviceName;
            this.diskReadKBytePerSec = diskReadKBytePerSec;
            this.diskWrittenKBytesPerSec = diskWrittenKBytesPerSec;
        }

        /**
         * Returns the name of the storage device
         * 
         * @return name of the storage device
         */
        public String getDeviceName() {
            return this.deviceName;
        }

        /**
         * Returns the number of kilobytes read per second
         * 
         * @return number of kilobytes read per second
         */
        public long getDiskReadKBytePerSec() {
            return this.diskReadKBytePerSec;
        }

        /**
         * Returns the number of kilobytes written per second
         * 
         * @return number of kilobytes written per second
         */
        public long getDiskWrittenKBytesPerSec() {
            return this.diskWrittenKBytesPerSec;
        }

        /**
         * Sets the device name
         * 
         * @param deviceName device name
         */
        public void setDeviceName(final String deviceName) {
            this.deviceName = deviceName;
        }

        /**
         * Sets the number of kilobytes read per second
         * 
         * @param diskReadKBytePerSec number of kilobytes read per second
         */
        public void setDiskReadKBytePerSec(final long diskReadKBytePerSec) {
            this.diskReadKBytePerSec = diskReadKBytePerSec;
        }

        /**
         * Sets the number of kilobytes written per second
         * 
         * @param diskWrittenKBytesPerSec number of kilobytes written per second
         */
        public void setDiskWrittenKBytesPerSec(final long diskWrittenKBytesPerSec) {
            this.diskWrittenKBytesPerSec = diskWrittenKBytesPerSec;
        }

        /*
         * (non-Javadoc)
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            return "(disk=" + this.deviceName + ",read=" + this.diskReadKBytePerSec + " KB/s,written="
                + this.diskWrittenKBytesPerSec + " KB/s)";
        }

    }

    /**
     * Network I/O performance metrics of a NIC
     */
    public static class NetworkStats implements Serializable {
        private static final long serialVersionUID = -815338799963686071L;

        private String deviceName;

        private long netReceivedKbitPerSec;

        private long netTransmittedKbitPerSec;

        /**
         * Constructs a new NetworkStats
         */
        public NetworkStats() {
        }

        /**
         * Constructs a new NetworkStats with specific parameters
         * 
         * @param deviceName name of the NIC
         * @param netReceivedKbitPerSec number of kilobits received per second
         * @param netTransmittedKbitPerSec number of kilobits transmitted per
         *        second
         */
        public NetworkStats(final String deviceName, final long netReceivedKbitPerSec, final long netTransmittedKbitPerSec) {
            super();
            this.deviceName = deviceName;
            this.netReceivedKbitPerSec = netReceivedKbitPerSec;
            this.netTransmittedKbitPerSec = netTransmittedKbitPerSec;
        }

        /**
         * Returns the name of the device
         * 
         * @return name of the device
         */
        public String getDeviceName() {
            return this.deviceName;
        }

        /**
         * Returns the number of kilobits received per second
         * 
         * @return number of kilobits received per second
         */
        public long getNetReceivedKbitPerSec() {
            return this.netReceivedKbitPerSec;
        }

        /**
         * Returns the number of kilobits transmitted per second
         * 
         * @return number of kilobits transmitted per second
         */
        public long getNetTransmittedKbitPerSec() {
            return this.netTransmittedKbitPerSec;
        }

        /**
         * Sets the device name
         * 
         * @param deviceName device name
         */
        public void setDeviceName(final String deviceName) {
            this.deviceName = deviceName;
        }

        /**
         * Sets the number of kilobits received per second
         * 
         * @param netReceivedKbitPerSec number of kilobits received per second
         */
        public void setNetReceivedKbitPerSec(final long netReceivedKbitPerSec) {
            this.netReceivedKbitPerSec = netReceivedKbitPerSec;
        }

        /**
         * Sets the number of kilobits transmitted per second
         * 
         * @param netTransmittedKbitPerSec number of kilobits transmitted per
         *        second
         */
        public void setNetTransmittedKbitPerSec(final long netTransmittedKbitPerSec) {
            this.netTransmittedKbitPerSec = netTransmittedKbitPerSec;
        }

        @Override
        public String toString() {
            return "(net=" + this.deviceName + ",rcv=" + this.netReceivedKbitPerSec + " Kb/s,tx="
                + this.netTransmittedKbitPerSec + " Kb/s)";
        }

    }

    private Date samplingTime;

    private float cpuLoad;

    private long memoryUsedKBytes;

    private List<NetworkStats> networkStats;

    private List<DiskStats> diskStats;

    /**
     * Constructs a new ResourceUsage
     */
    public ResourceUsage() {
    }

    /**
     * Constructs a new ResourceUsage from a CompositeDate object
     * 
     * @param cd composite data representing a ResourceUsage
     * @return a ResourceUsage object represented by cd
     */
    public static ResourceUsage from(final CompositeData cd) {
        ResourceUsage ru = new ResourceUsage();
        ru.setSamplingTime((Date) cd.get("samplingTime"));
        ru.setCpuLoad((Float) cd.get("cpuLoad"));
        ru.setMemoryUsedKBytes((Long) cd.get("memoryUsedKBytes"));

        CompositeData[] cds = (CompositeData[]) cd.get("networkStats");
        ArrayList<ResourceUsage.NetworkStats> netStats = new ArrayList<ResourceUsage.NetworkStats>();
        if (cds != null) {
            for (CompositeData ncd : cds) {
                ResourceUsage.NetworkStats entry = new ResourceUsage.NetworkStats();
                entry.setDeviceName((String) ncd.get("deviceName"));
                entry.setNetReceivedKbitPerSec((Long) ncd.get("netReceivedKbitPerSec"));
                entry.setNetTransmittedKbitPerSec((Long) ncd.get("netTransmittedKbitPerSec"));
                netStats.add(entry);
            }
        }
        ru.setNetworkStats(netStats);

        cds = (CompositeData[]) cd.get("diskStats");
        ArrayList<ResourceUsage.DiskStats> diskStats = new ArrayList<ResourceUsage.DiskStats>();
        if (cds != null) {
            for (CompositeData ncd : cds) {
                ResourceUsage.DiskStats entry = new ResourceUsage.DiskStats();
                entry.setDeviceName((String) ncd.get("deviceName"));
                entry.setDiskReadKBytePerSec((Long) ncd.get("diskReadKBytePerSec"));
                entry.setDiskWrittenKBytesPerSec((Long) ncd.get("diskWrittenKBytesPerSec"));
                diskStats.add(entry);
            }
        }
        ru.setDiskStats(diskStats);
        return ru;
    }

    /**
     * Returns the time at which the ResourceUsage was sampled
     * 
     * @return the time at which the ResourceUsage was sampled
     */
    public Date getSamplingTime() {
        return this.samplingTime;
    }

    /**
     * Returns the CPU load of the VM as the fraction of of the physical CPU
     * capacity
     * 
     * @return the CPU load (between 0 and 1)
     */
    public float getCpuLoad() {
        return this.cpuLoad;
    }

    /**
     * Returns the memory occupation of the VM in kilobytes
     * 
     * @return memory occupation of the VM in kilobytes
     */
    public long getMemoryUsedKBytes() {
        return this.memoryUsedKBytes;
    }

    /**
     * Returns the network performance statistics of all NICs attached to the VM
     * 
     * @return list of NetworkStats representing the network performance of all
     *         NICs attached to the VM
     */
    public List<NetworkStats> getNetworkStats() {
        return this.networkStats;
    }

    /**
     * Returns the the disk I/O performance statistics of all disk devices
     * attached to the VM
     * 
     * @return the disk I/O performance statistics of all disk devices attached
     *         to the VM
     */
    public List<DiskStats> getDiskStats() {
        return this.diskStats;
    }

    /**
     * Sets the time at which the ResourceUsage is sampled
     * 
     * @param samplingTime time at which the ResourceUsage is sampled
     */
    public void setSamplingTime(final Date samplingTime) {
        this.samplingTime = samplingTime;
    }

    /**
     * Sets the CPU load of the VM as the fraction of of the physical CPU
     * capacity
     * 
     * @param cpuLoad the CPU load
     */
    public void setCpuLoad(final float cpuLoad) {
        this.cpuLoad = cpuLoad;
    }

    /**
     * Sets the memory occupation of the VM in kilobytes
     * 
     * @param memoryUsedKBytes memory occupation of the VM in kilobytes
     */
    public void setMemoryUsedKBytes(final long memoryUsedKBytes) {
        this.memoryUsedKBytes = memoryUsedKBytes;
    }

    /**
     * Sets the network performance statistics of all NICs attached to the VM
     * 
     * @param networkStats list of NetworkStats representing the network
     *        performance of all NICs attached to the VM
     */
    public void setNetworkStats(final List<NetworkStats> networkStats) {
        this.networkStats = networkStats;
    }

    /**
     * Sets the disk I/O performance statistics of all disk devices attached to
     * the VM
     * 
     * @param diskStats list of DiskStats representing the disk I/O performance
     *        of all disk devices attached to the VM
     */
    public void setDiskStats(final List<DiskStats> diskStats) {
        this.diskStats = diskStats;
    }

    @Override
    public String toString() {
        StringBuffer sb = new StringBuffer("(time=" + this.samplingTime + "),(cpu=" + this.cpuLoad * 100 + "%),(memory="
            + this.memoryUsedKBytes + " Kbytes),");
        if (this.diskStats != null) {
            for (DiskStats ds : this.diskStats) {
                sb.append(ds);
                sb.append(",");
            }
        }
        if (this.networkStats != null) {
            for (NetworkStats ns : this.networkStats) {
                sb.append(ns);
                sb.append(",");
            }
        }
        return sb.toString();
    }

}
