/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerPoolMXBean.java 6064 2010-02-18 14:05:50Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.api;

import java.util.List;
import java.util.Map;

/**
 * Management interface of a Server Pool. A Server Pool is a management unit
 * acting as a container of physical hosts and virtual machines. A Server Pool
 * is homogeneous with regard to the virtualization technology used by its
 * member hosts. Live or cold migration of virtual machines between hosts is
 * allowed only if the source and target host belong to the same server pool.
 * <p>
 * The <tt>ObjectName</tt> for identifying a ServerPoolMXBean follows the
 * following template: <blockquote>
 * org.ow2.jasmine.vmm.api:type=ServerPool,name=&lt;pathname&gt; </blockquote>
 * <p>
 */
public interface ServerPoolMXBean extends ManagedResourceMXBean {
    /**
     * Return the symbolic name of the server pool
     *
     * @return
     */
    String getName();

    /**
     * Provisions a new virtual machine. The host on which the VM will be
     * created will depend on the provisioning policy active for Server Pool.
     * Upon successful creation a notification of type NotificationTypes.VM_NEW
     * will be emitted by the HostMXBean on which the VM has been created.
     *
     * @param vmSpec initial configuration of the virtual machine
     * @param constraints constraints on the placement of the VM
     * @param sync if true the call will return once the VM has been created
     * @return the management interface of the created VM if <tt>sync</tt> is
     *         true, null otherwise
     * @throws InsufficientResourcesException raised if not enough resources are
     *         available to create the VM
     * @throws InvalidVMConfigException raised if the VM configuration is
     *         invalid
     */
    VirtualMachineMXBean provisionVM(VMConfigSpec vmSpec, Map<String, String> constraints, boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException;

    /**
     * Sets the provisioning policy of this domain
     *
     * @param policyId
     */
    void setProvisioningPolicy(String policyId);

    /**
     * /** Returns the physical hosts belong to the server pool
     *
     * @return
     */
    List<HostMXBean> getManagedHosts();

    /**
     * Returns the image stores belonging to this server pool
     *
     * @return
     */
    VirtualMachineImageStoreMXBean getVMImageStore();

}
