/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMConfigSpec.java 6064 2010-02-18 14:05:50Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.api;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The <tt>VMConfigSpec</tt> class is a helper class used to describe the
 * configuration of a virtual machine to be created
 */
public class VMConfigSpec implements Serializable {

    /**
     * Auto-generated serial version ID.
     */
    private static final long serialVersionUID = -5174963382169498530L;

    /**
     * Constructs a <tt>VMConfigSpec</tt> object
     * 
     * @param name the name of the virtual machine
     * @param memorySizeMB the initial memory allocation in MBytes
     * @param numVCPU the number of virtual CPUs
     * @param diskSizeMB the disk size in MBytes
     * @param vmImageUUID the uuid of the VM image that will be used to create
     *        the VM
     */
    public VMConfigSpec(final String name, final long memorySizeMB, final int numVCPU, final int diskSizeMB,
        final String vmImageUUID) {
        this();
        this.name = name;
        this.memorySizeMB = memorySizeMB;
        this.numVCPU = numVCPU;
        this.diskSizeMB = diskSizeMB;
        this.vmImageUUID = vmImageUUID;
    }

    /**
     * Constructs a <tt>VMConfigSpec</tt> object as a copy of the argument
     * 
     * @param other original object to copy
     */
    public VMConfigSpec(final VMConfigSpec other) {
        this.name = new String(other.name);
        this.memorySizeMB = other.memorySizeMB;
        this.numVCPU = other.numVCPU;
        this.diskSizeMB = other.diskSizeMB;
        this.vmImageUUID = new String(other.vmImageUUID);
        this.guestOsDomain = other.guestOsDomain;
        this.guestOsHostName = other.guestOsHostName;
        this.guestOsDnsServerList = other.guestOsDnsServerList != null ? new ArrayList<String>(other.guestOsDnsServerList)
            : null;
        this.vnicIpSettingsList = other.vnicIpSettingsList != null ? new ArrayList<VnicIPSettings>(other.vnicIpSettingsList)
            : null;
        this.guestOsCustomizationParams = other.guestOsCustomizationParams != null ? new HashMap<String, String>(
            other.guestOsCustomizationParams) : null;
    }

    /**
     * Constructs a <tt>VMConfigSpec</tt> object
     */
    public VMConfigSpec() {
        this.guestOsDnsServerList = new ArrayList<String>();
        this.vnicIpSettingsList = new ArrayList<VnicIPSettings>();
        this.guestOsCustomizationParams = new HashMap<String, String>();
    }

    private String name;

    private long memorySizeMB;

    private int numVCPU;

    private int diskSizeMB;

    private String vmImageUUID;

    private String guestOsDomain;

    private String guestOsHostName;

    private List<String> guestOsDnsServerList;

    private List<VnicIPSettings> vnicIpSettingsList;

    private Map<String, String> guestOsCustomizationParams;

    /**
     * Sets the name of the virtual machine
     * 
     * @param name
     */
    public void setName(final String name) {
        this.name = name;
    }

    /**
     * Sets the memory size of the virtual machine configuration in MBytes
     * 
     * @param memorySizeMB
     */
    public void setMemorySizeMB(final long memorySizeMB) {
        this.memorySizeMB = memorySizeMB;
    }

    /**
     * Sets the number of virtual CPUs
     * 
     * @param numVCPU
     */
    public void setNumVCPU(final int numVCPU) {
        this.numVCPU = numVCPU;
    }

    /**
     * Sets the disk size in MBytes
     * 
     * @param diskSizeMB
     */
    public void setDiskSizeMB(final int diskSizeMB) {
        this.diskSizeMB = diskSizeMB;
    }

    /**
     * Sets the image ID of the virtual machine
     * 
     * @param diskImageID
     */
    public void setVmImageUUID(final String diskImageID) {
        this.vmImageUUID = diskImageID;
    }

    /**
     * Returns the name of virtual machine
     * 
     * @return name of the virtual machine
     */
    public String getName() {
        return this.name;
    }

    /**
     * Returns the memory size of the virtual machine in MBytes
     * 
     * @return
     */
    public long getMemorySizeMB() {
        return this.memorySizeMB;
    }

    /**
     * Returns the number of virtual CPUs of the virtual machine configuration
     * 
     * @return
     */
    public int getNumVCPU() {
        return this.numVCPU;
    }

    /**
     * Returns the disk size in MBytes of the virtual machine configuration
     * 
     * @return
     */
    public int getDiskSizeMB() {
        return this.diskSizeMB;
    }

    /**
     * Returns the image ID of the virtual machine configuration
     * 
     * @return
     */
    public String getVmImageUUID() {
        return this.vmImageUUID;
    }

    /**
     * Sets the DNS domain the guest OS belongs to
     * 
     * @param guestOsDomain DNS domain name
     */
    public void setGuestOsDomain(final String domain) {
        this.guestOsDomain = domain;
    }

    /**
     * Returns the guest OS DNS domain
     * 
     * @return
     */
    public String getGuestOsDomain() {
        return this.guestOsDomain;
    }

    /**
     * Returns the non-qualified hostname of the guest OS
     * 
     * @return
     */
    public String getGuestOsHostName() {
        return this.guestOsHostName;
    }

    /**
     * Sets the non-qualified hostname of the guest OS
     * 
     * @param guestOsHostName non-qualified hostname
     */
    public void setGuestOsHostName(final String hostName) {
        this.guestOsHostName = hostName;
    }

    /**
     * Returns the list of DNS servers to be used by the guest OS
     * 
     * @return list of DNS servers
     */
    public List<String> getGuestOsDnsServerList() {
        return new ArrayList<String>(this.guestOsDnsServerList);
    }

    /**
     * Sets the list of DNS servers to be used by the guest OS
     * 
     * @param guestOsDnsServerList list of DNS servers
     */
    public void setGuestOsDnsServerList(final List<String> dnsServerList) {
        this.guestOsDnsServerList = new ArrayList<String>(dnsServerList);
    }

    /**
     * Adds an IP settings associated with a NIC attached to the VM
     * 
     * @param ipSettings
     */
    public void addVnicIpSettings(final VnicIPSettings ipSettings) {
        this.vnicIpSettingsList.add(ipSettings);
    }

    /**
     * Sets the VNIC IP settings of the VNIC(s) attached to this VM
     * 
     * @param ipSettingsList the list of VNIC IP settings
     */
    public void setVnicIpSettingsList(final List<VnicIPSettings> ipSettingsList) {
        this.vnicIpSettingsList = ipSettingsList;
    }

    /**
     * Returns the IP settings of each NIC attached to the VM
     * 
     * @return IP settings of each NIC attached to the VM
     */
    public List<VnicIPSettings> getVnicIpSettingsList() {
        return new ArrayList<VnicIPSettings>(this.vnicIpSettingsList);
    }

    /**
     * Returns the guest OS specific customization parameters
     * 
     * @return the customization parameters as a map of (key, value)
     */
    public Map<String, String> getGuestOsCustomizationParams() {
        return this.guestOsCustomizationParams;
    }

    /**
     * Sets guest-OS specific customization parameters
     * 
     * @param guestCustomization the list of customization parameters to set
     */
    public void setGuestOsCustomizationParams(final Map<String, String> guestCustomizationParams) {
        this.guestOsCustomizationParams = guestCustomizationParams;
    }

    /**
     * Adds a guest-OS specific customization parameters
     * 
     * @param paramName key of the parameter to add
     * @param paramValue value of the parameter to add
     */
    public void addGuestOsCustomizationParam(final String paramName, final String paramValue) {
        this.guestOsCustomizationParams.put(paramName, paramValue);
    }

}
