/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * $Id: VMCustomizationSpec.java 6064 2010-02-18 14:05:50Z jlegrand $
 */
package org.ow2.jasmine.vmm.api;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The <tt>VMCustomizationSpec</tt> class is a helper class used to describe the
 * customization parameters of a virtual machine to be created or cloned
 */
public class VMCustomizationSpec implements Serializable {
    private static final long serialVersionUID = -8000190535943480037L;

    private String guestOsDomain;

    private String guestOsHostName;

    private List<String> guestOsDnsServerList;

    private List<VnicIPSettings> vnicIpSettingsList;

    private Map<String, String> guestOsCustomizationParams;

    /**
     * Constructs a <tt>VMConfigSpec</tt> object as a copy of the argument
     * 
     * @param other original object to copy
     */
    public VMCustomizationSpec(final VMCustomizationSpec other) {
        this.guestOsDomain = other.guestOsDomain;
        this.guestOsHostName = other.guestOsHostName;
        this.guestOsDnsServerList = other.guestOsDnsServerList != null ? new ArrayList<String>(other.guestOsDnsServerList)
            : null;
        this.vnicIpSettingsList = other.vnicIpSettingsList != null ? new ArrayList<VnicIPSettings>(other.vnicIpSettingsList)
            : null;
        this.guestOsCustomizationParams = other.guestOsCustomizationParams != null ? new HashMap<String, String>(
            other.guestOsCustomizationParams) : null;
    }

    /**
     * Constructs a <tt>VMConfigSpec</tt> object
     */
    public VMCustomizationSpec() {
        this.guestOsDnsServerList = new ArrayList<String>();
        this.vnicIpSettingsList = new ArrayList<VnicIPSettings>();
        this.guestOsCustomizationParams = new HashMap<String, String>();
    }

    /**
     * Sets the DNS domain the guest OS belongs to
     * 
     * @param guestOsDomain DNS domain name
     */
    public void setGuestOsDomain(final String domain) {
        this.guestOsDomain = domain;
    }

    /**
     * Returns the guest OS DNS domain
     * 
     * @return
     */
    public String getGuestOsDomain() {
        return this.guestOsDomain;
    }

    /**
     * Returns the non-qualified hostname of the guest OS
     * 
     * @return
     */
    public String getGuestOsHostName() {
        return this.guestOsHostName;
    }

    /**
     * Sets the non-qualified hostname of the guest OS
     * 
     * @param guestOsHostName non-qualified hostname
     */
    public void setGuestOsHostName(final String hostName) {
        this.guestOsHostName = hostName;
    }

    /**
     * Returns the list of DNS servers to be used by the guest OS
     * 
     * @return list of DNS servers
     */
    public List<String> getGuestOsDnsServerList() {
        return new ArrayList<String>(this.guestOsDnsServerList);
    }

    /**
     * Sets the list of DNS servers to be used by the guest OS
     * 
     * @param guestOsDnsServerList list of DNS servers
     */
    public void setGuestOsDnsServerList(final List<String> dnsServerList) {
        this.guestOsDnsServerList = new ArrayList<String>(dnsServerList);
    }

    /**
     * Adds an IP settings associated with a NIC attached to the VM
     * 
     * @param ipSettings
     */
    public void addVnicIpSettings(final VnicIPSettings ipSettings) {
        this.vnicIpSettingsList.add(ipSettings);
    }

    /**
     * Sets the VNIC IP settings of the VNIC(s) attached to this VM
     * 
     * @param ipSettingsList the list of VNIC IP settings
     */
    public void setVnicIpSettingsList(final List<VnicIPSettings> ipSettingsList) {
        this.vnicIpSettingsList = ipSettingsList;
    }

    /**
     * Returns the IP settings of each NIC attached to the VM
     * 
     * @return IP settings of each NIC attached to the VM
     */
    public List<VnicIPSettings> getVnicIpSettingsList() {
        return new ArrayList<VnicIPSettings>(this.vnicIpSettingsList);
    }

    /**
     * Returns the guest OS specific customization parameters
     * 
     * @return the customization parameters as a map of (key, value)
     */
    public Map<String, String> getGuestOsCustomizationParams() {
        return this.guestOsCustomizationParams;
    }

    /**
     * Sets guest-OS specific customization parameters
     * 
     * @param guestCustomization the list of customization parameters to set
     */
    public void setGuestOsCustomizationParams(final Map<String, String> guestCustomizationParams) {
        this.guestOsCustomizationParams = guestCustomizationParams;
    }

    /**
     * Adds a guest-OS specific customization parameters
     * 
     * @param paramName key of the parameter to add
     * @param paramValue value of the parameter to add
     */
    public void addGuestOsCustomizationParam(final String paramName, final String paramValue) {
        this.guestOsCustomizationParams.put(paramName, paramValue);
    }

}
