/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: NotSecuredApplicationClient.java 13538 2008-04-07 12:01:32Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.examples.ear.client;

import java.io.PrintStream;
import java.util.Collection;
import java.util.List;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.ow2.jonas.examples.ear.entity.Author;
import org.ow2.jonas.examples.ear.entity.Book;
import org.ow2.jonas.examples.ear.init.Initializer;
import org.ow2.jonas.examples.ear.mail.Mailer;
import org.ow2.jonas.examples.ear.reader.RemoteReader;

/**
 * Simple Application Client.
 * @author Guillaume Sauthier
 */
public final class NotSecuredApplicationClient {

    /**
     * Empty default constructor for utility class.
     */
    private NotSecuredApplicationClient() {

    }

    /**
     * @param args Command line arguments
     * @throws NamingException InitialContext creation failure
     */
    public static void main(final String[] args) throws NamingException {

        PrintStream out = System.out;

        // Print Header
        out.println("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - -");
        out.println("OW2 JOnAS :: EAR Sample :: Not Secured Application Client  ");
        out.println("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - -");

        // Lookup the Remote Bean interface through JNDI
        Context initialContext = new InitialContext();

        // Init datas if needed
        out.print("Initialization ... ");
        Initializer bean = (Initializer) initialContext.lookup("myInitializerBean");
        bean.initializeEntities();
        out.println("Done.");

        // Get the bean
        String jndiName = "org.ow2.jonas.examples.ear.reader.ReaderBean"
                          + "_" + RemoteReader.class.getName() + "@Remote";
        RemoteReader readerBean = (RemoteReader) initialContext.lookup(jndiName);
        out.println("Get the RemoteReader Bean reference: " + readerBean);

        // List registered authors and their books
        out.println("List of authors, and their books:");
        // retrieve again the authors list (now it has been initialized)
        List<Author> authors = readerBean.listAllAuthors();
        for (Author author : authors) {
            out.println(" * " + author.getName());
            Collection<Book> books = author.getBooks();
            for (Book book : books) {
                out.println(" -> " + book.getTitle() + " [id: " + book.getId() + "]");
            }
        }

        // Only list registered books
        List<Book> books = readerBean.listAllBooks();
        out.println("List of books:");
        for(Book book : books) {
            out.println(" * " + book.getTitle() + " [" + book.getAuthor().getName() + "]");
        }

        // Use the Mailer bean to send
        // Use the first command line argument as a mail address.
        // Fall back to a reasonable default ${user.name}@localhost
        String address = System.getProperty("user.name") + "@localhost";
        if (args.length > 0) {
            // Got an argument, use it ...
            address = args[0];
        }

        // Get the Mailer Bean reference
        Mailer mailerBean = (Mailer) initialContext.lookup("myMailerBean");
        out.println("Get the Mailer Bean reference: " + mailerBean);

        // Call the Mailer bean to send the expected e-mail.
        mailerBean.sendStatusMail(address);

        // OK, we're done
        out.println("Success.");

    }
}
