/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: JMSApplicationClient.java 15223 2008-09-25 14:31:07Z coqp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.examples.ear.client;

import java.io.PrintStream;
import java.util.Collection;
import java.util.List;
import java.text.MessageFormat;

import javax.jms.*;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.ejb.EJB;
import javax.annotation.Resource;

import org.ow2.jonas.examples.ear.entity.Author;
import org.ow2.jonas.examples.ear.entity.Book;
import org.ow2.jonas.examples.ear.init.Initializer;
import org.ow2.jonas.examples.ear.reader.RemoteReader;

/**
 * This application-client shows usage of JMS destinations to
 * interact with the server-side application.
 * @author Guillaume Sauthier
 */
public final class JMSApplicationClient {

    /**
     * Number of Books to be created.
     */
    private static final int ITERATION_NUMBER = 10;

    /**
      * Link to the initializer bean.
      */
     @EJB
     private static  Initializer initializerBean;

    /**
     *  JMS conectionFactoery
     */
    // Resource injection will be soon implemented
     @Resource(mappedName="JQCF")
      private static ConnectionFactory factory;

    /**
     *   JMS Queue  SampleQueue
     */
    // Resource injection Will be soon implemented
     @Resource(mappedName="SampleQueue")
     private static Queue queue;


    /**
      * Link to the Remote Reader bean. Bean will be injected by JOnAS.
      */
     @EJB
     static private RemoteReader readerBean;

    /**
     * Empty default constructor for utility class.
     */
    private JMSApplicationClient() {

    }

    /**
     * @param args Command line arguments
     * @throws Exception InitialContext creation failure / JMS Exception
     */
    public static void main(final String[] args) throws Exception {

        PrintStream out = System.out;

        // Print Header
        out.println("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - -");
        out.println("OW2 JOnAS :: EAR Sample :: Messager Application Client     ");
        out.println("- - - - - - - - - - - - - - - - - - - - - - - - - - - - - -");




        // Init. data if needed
        out.print("Initialization ... ");

        initializerBean.initializeEntities();
        out.println("Done.");
        // Lookup the Remote Bean interface through JNDI
        Context initialContext = new InitialContext();
        // Get the JMS QueueConnectionFactory
        factory = (QueueConnectionFactory) initialContext.lookup("java:comp/env/jms/QueueConnectionFactory");

        // Get the JMS Queue
        queue = (Queue) initialContext.lookup("java:comp/env/jms/SampleQueue");

        // Send Book creation Messages
        Connection connection = factory.createConnection();
        Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
        MessageProducer messageProducer= session.createProducer(queue);

        for (int i = 0; i < ITERATION_NUMBER; i++) {
            String title = "Encyclopedia Universalis Vol." + i;
            Message message = session.createTextMessage(title);
            messageProducer.send(message);
            out.println("Sended creation order for '" + title + "'");
        }

        // Close JMS objects
        messageProducer.close();
        session.close();
        connection.close();

        // Wait for some time ...
        // Remember JMS is for asynchronous messages :)
        final long period = 2500;
        out.println(MessageFormat.format("Wait for {0} ms...", period));
        Thread.sleep(period);

        out.println(MessageFormat.format("The RemoteReader Bean reference is: {0}", readerBean));
        // List Authors and Books
        List<Author> authors = readerBean.listAllAuthors();
        for (Author author : authors) {
            out.println(MessageFormat.format(" * {0}", author.getName()));
            Collection<Book> books = author.getBooks();
            for (Book book : books) {
                out.println(MessageFormat.format(" -> {0} [id: {1}]", book.getTitle(), book.getId()));
            }
        }

        out.println("Success.");

    }
}
