/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: OSGiClassLoader.java 15741 2008-11-12 16:35:33Z eyindanga $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.lib.loader;

import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;

/**
 * {@link OSGiClassLoader} delegates class loading to the OSGi framework.
 * @author Francois Fornaciari
 */
public class OSGiClassLoader extends URLClassLoader {

    /**
     * This bundle's Bundle ClassLoader.
     */
    private ClassLoader classLoader = null;

    /**
     * Construct a new delegating ClassLoader.
     */
    public OSGiClassLoader() {
        super(new URL[0]);
        this.classLoader = getClass().getClassLoader();
    }

    /**
     * {@inheritDoc}
     * @see java.lang.ClassLoader#loadClass(java.lang.String)
     */
    @Override
    public Class<?> loadClass(final String name) throws ClassNotFoundException {
        return classLoader.loadClass(name);
    }

    /**
     * {@inheritDoc}
     * @see java.lang.ClassLoader#loadClass(java.lang.String, boolean)
     */
    @Override
    public Class<?> loadClass(final String name, final boolean flag) throws ClassNotFoundException {
        return loadClass(name);
    }

    /**
     * {@inheritDoc}
     * @see java.lang.ClassLoader#getResource(java.lang.String)
     */
    @Override
    public URL getResource(final String name) {
        // First look if OSGi can find the resource we want
        URL resource = null;
        try {
            resource = classLoader.getResource(name);
        } catch (IllegalArgumentException e) {
         // Fall-back: if not found, try the usual getResource() search mechanism.
            // This will let our services to access properties file that are in JB/conf
            try {
                resource = super.getResource(name);
            } catch (IllegalArgumentException e2) {
                return resource;
            }
                
        }
        if (resource == null ) {
            resource = super.getResource(name);
        }
        // Finally return the resource (may be null)
        return resource;
    }

    /**
     * {@inheritDoc}
     * @see java.lang.ClassLoader#getResourceAsStream(java.lang.String)
     */
    @Override
    public InputStream getResourceAsStream(final String name) {
        // First look if OSGi can find the resource we want
        InputStream is = classLoader.getResourceAsStream(name);

        // Fall-back: if not found, try the usual getResourceAsStream() search mechanism.
        // This will let our services to access properties file that are in JB/conf
        if (is == null) {
            is = super.getResourceAsStream(name);
        }

        // Finally return the resource (may be null)
        return is;
    }

}
