/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2004 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: BeanNaming.java 15428 2008-10-07 11:20:29Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.lib.util;

import java.io.File;
import java.lang.reflect.Method;
import java.util.StringTokenizer;

import javax.security.jacc.EJBMethodPermission;

/**
 * This class is made for hiding Naming convention in all JOnAS framework. <br>
 * (Among other things, names of the implementation classes of the Enterprise
 * Bean's Home and Enterprise Bean Remote interfaces generated by the GenIC
 * tools.) <br>
 * It is used by the EJB generation tools and framework
 *
 * @author Helene Joanin
 * @author Joe Gittings has proposed to code method signature in order to avoid same signature for inherited methods
 * @author Christophe Ney [cney@batisseurs.com]
 * @author Sebastien Chassande-Barrioz
 */
public class BeanNaming {

    /**
     * returns the name of the package of the given full name
     */
    public static String getPackageName(String name) {
        if (name == null) { return null; }
        int idx = name.lastIndexOf('.');
        return (idx == -1 ? "" : name.substring(0, idx));
    }

    /**
     * return full name of a class for given package and class names
     *
     * @param packageName
     *            possibly empty package name
     * @param name
     *            class name
     * @return fully qualified class name
     */
    public static String getClassName(String packageName, String name) {
        return (packageName.length() == 0) ? name : packageName + "." + name;
    }

    /**
     * returns the basename of the given full name
     */
    public static String getBaseName(String fullName) {
        String baseName = null;
        if (fullName != null) {
            int idx = fullName.lastIndexOf('.');
            int len = fullName.length();
            if (idx == -1) {
                baseName = fullName;
            } else {
                if (idx != (len - 1)) {
                    baseName = fullName.substring(idx + 1, len);
                } else {
                    baseName = "";
                }
            }
        }
        return baseName;
    }

    /**
     * returns the full path of the file name. mainly replace '.' by '/'.
     */
    public static String getPath(String fullName) {
        String pkg = new String();
        StringTokenizer stk = new StringTokenizer(fullName, ".");
        int nb = stk.countTokens();
        for (int i = 0; i < nb; i++) {
            pkg = pkg.concat(stk.nextToken());
            if (i < nb - 1) {
                pkg = pkg + File.separatorChar;
            }
        }
        return pkg;
    }

    /**
     * returns the given string with the first character converted to upper case
     */
    public static String firstToUpperCase(String s) {
        String value;
        if (s.length() > 0) {
            char c = Character.toUpperCase(s.charAt(0));
            value = new String(c + s.substring(1));
        } else {
            value = new String();
        }
        return (value);
    }

    /**
     * returns the name of the JOnAS specific deployment descriptor file's name
     * corresponding to the given standard deployment descriptor file's name.
     * (ex: returns "jonas-XXX.xml" for "XXX.xml" and "../../jonas-XXX.xml" for
     * "../../XXX.xml")
     */
    public static String getJonasXmlName(String stdXmlName) {

        String jonasXmlName = new String();
        File f = new File(stdXmlName);
        String p = f.getParent();
        String n = f.getName();
        if (p != null) {
            jonasXmlName = jonasXmlName.concat(p);
            jonasXmlName = jonasXmlName.concat(File.separator);
        }
        jonasXmlName = jonasXmlName.concat("jonas-");
        jonasXmlName = jonasXmlName.concat(n);

        return (jonasXmlName);
    }

    /**
     * returns the name of the parent directory
     */
    public static String getParentName(String fileName) {
        File f = new File(fileName);
        return f.getParent();
    }

    /**
     * Gets a string that represents the signature of a method
     *
     * @param ejbName
     *            name of the ejb
     * @param method
     *            Method on which the signature is required
     * @return string that represents the signature of a method
     */
    public static String getSignature(String ejbName, Method method) {

        Class clazz = method.getDeclaringClass();
        String methItf = "";

        if (javax.ejb.EJBHome.class.isAssignableFrom(clazz)) {
            methItf = "Home";
        } else if (javax.ejb.EJBObject.class.isAssignableFrom(clazz)) {
            methItf = "Remote";
        } else if (javax.ejb.EJBLocalHome.class.isAssignableFrom(clazz)) {
            methItf = "LocalHome";
        } else if (javax.ejb.EJBLocalObject.class.isAssignableFrom(clazz)) {
            methItf = "Local";
        } else if (java.rmi.Remote.class.isAssignableFrom(clazz)) {
            methItf = "ServiceEndpoint";
        }

        return new EJBMethodPermission(ejbName, methItf, method).getActions();
    }
}

