/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: WeldContextCustomizer.java 19397 2010-03-09 14:58:31Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.cdi.weld.internal.tomcat;

import org.apache.catalina.LifecycleListener;
import org.apache.catalina.core.StandardContext;
import org.jboss.weld.bootstrap.spi.BeanDeploymentArchive;
import org.jboss.weld.bootstrap.spi.Deployment;
import org.jboss.weld.servlet.WeldListener;
import org.jboss.weld.servlet.api.ServletServices;
import org.ow2.jonas.cdi.weld.IDeploymentBuilder;
import org.ow2.jonas.cdi.weld.IWeldService;
import org.ow2.jonas.cdi.weld.internal.WeldDeployableInfo;
import org.ow2.jonas.cdi.weld.internal.services.JOnASServletServices;
import org.ow2.jonas.web.tomcat6.custom.ContextCustomizer;
import org.ow2.util.ee.deploy.api.deployable.WARDeployable;

/**
 * A {@code WeldContextCustomizer} will add to the Tomcat Context the appropriate
 * application and Context listeners.
 *
 * @author Guillaume Sauthier
 */
public class WeldContextCustomizer implements ContextCustomizer {

    /**
     * Our weld service.
     */
    private IWeldService weld;

    /**
     * Our Deployment factory.
     */
    private IDeploymentBuilder builder;

    /**
     * Bind the WeldService reference.
     */
    public void setWeldService(final IWeldService weld) {
        this.weld = weld;
    }

    /**
     * Bind the builder.
     */
    public void setDeploymentBuilder(final IDeploymentBuilder builder) {
        this.builder = builder;
    }

    /**
     * Customize the given Catalina Context instance using the given WAR deployable.
     * This is called just before the Context start-up.
     *
     * @param context    Catalina Context to be customized
     * @param deployable the WAR being deployed
     */
    public void customize(final StandardContext context, final WARDeployable deployable) {

        // Currently, only standalone applications have their WARDeployable passed through services
        // If no deployable is here, just return
        // TODO Should be removed when CDI support will be wider (ejbjar, ear, ...)
        if (deployable == null) {
            return;
        }

        // Only handle CDI enabled web applications
        if (weld.isCdiEnabled(deployable.getArchive())) {

            WeldDeployableInfo info = (WeldDeployableInfo) deployable.getExtension(WeldDeployableInfo.class);
            Deployment deployment;
            if (info == null) {

                // This is a standalone web-app

                // Create and store the extension info
                info = new WeldDeployableInfo();
                deployable.addExtension(info);

                // Build the Deployment from the archive
                deployment = builder.buildWebDeployment(deployable.getArchive());

                // Provides the Servlet Services
                // This is used in the WeldListener to retrieve the right Weld instance
                BeanDeploymentArchive main = deployment.getBeanDeploymentArchives().iterator().next();
                deployment.getServices().add(ServletServices.class, new JOnASServletServices(main));

                // Add a Tomcat LifecycleListener to hook up during Context lifecycle events
                LifecycleListener listener = new WeldLifeCycleListener(weld, deployment, info);
                context.addLifecycleListener(listener);
            }

            // Add an application listener that handle scope
            // boundaries (application, session, request)
            context.addApplicationListener(WeldListener.class.getName());

        }
    }
}
