/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2004 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * Initial developer(s): Florent BENOIT
 * --------------------------------------------------------------------------
 * $Id: AbsDBServiceImpl.java 21082 2011-03-31 08:58:05Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.db.base;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;
import org.ow2.jonas.lib.service.AbsServiceImpl;
import org.ow2.jonas.lib.util.Log;
import org.ow2.jonas.service.ServiceException;

/**
 * Abstract database service to be implemented by implementation of Java
 * databases.
 * @author Florent Benoit
 */
public abstract class AbsDBServiceImpl extends AbsServiceImpl {

    /**
     * Property for the default database name.
     */
    private static final String DEFAULT_DATABASE_NAME = "db_jonas";

    /**
     * Logger used by this service.
     */
    private static Logger logger = Log.getLogger(Log.JONAS_DB_PREFIX);

    /**
     * Users' List.
     */
    private List<User> users;

    /**
     * DB port (null means default).
     */
    private String port = null;

    /**
     * Database name.
     */
    private String dbName = DEFAULT_DATABASE_NAME;

    /**
     * Constructor initialization.
     */
    public AbsDBServiceImpl() {
        // Init user list
        this.users = new ArrayList<User>();
    }

    /**
     * Transform the pair user:pass into User instances.
     * @param users String of user/password (separated by a ":")
     */
    public void setUsers(final String users) {
        List<String> configuredUsers = convertToList(users);

        // Convert String user to real User.
        for (String user : configuredUsers) {
            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "Adding user/password '" + user + "'.");
            }

            StringTokenizer st = new StringTokenizer(user, ":");
            String name = st.nextToken();
            String pass = "";
            if (st.hasMoreTokens()) {
                pass = st.nextToken();
            }
            this.users.add(new User(name, pass));
        }

    }

    /**
     * @param port Database port
     */
    public void setPort(final String port) {
        this.port = port;
    }

    /**
     * @param db Database name
     */
    public void setDbname(final String db) {
        this.dbName = db;
    }

    /**
     * Create a database with the specified arguments. Need to be implemented by
     * classes extending this one.
     * @param users user/password (separated by a ":")
     * @param databaseName name of the database
     * @param portNumber port number of the database
     */
    protected abstract void initServer(List<User> users, String databaseName, String portNumber) throws ServiceException;

    /**
     * Start the service.
     * @throws ServiceException if the startup failed.
     */
    @Override
    protected void doStart() throws ServiceException {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "");
        }

        // Start the DB server
        initServer(users, dbName, port);
    }

    /**
     * Stop the service.
     * @throws ServiceException if the stop failed.
     */
    @Override
    protected void doStop() throws ServiceException {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "");
        }
    }

    /**
     * @return the logger.
     */
    public static Logger getLogger() {
        return logger;
    }

}
