/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:RegistryServiceImpl.java 10372 2007-05-14 13:58:42Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.configadmin.internal;

import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.ow2.jonas.configadmin.internal.model.Configadmin;
import org.ow2.jonas.configadmin.internal.model.ConfigurationType;
import org.ow2.util.archive.api.IFileArchive;
import org.testng.annotations.Test;
import org.testng.annotations.BeforeTest;

import java.net.URL;
import java.util.Dictionary;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

public class ConfigAdminDeployerTestCase {

    private ConfigAdminDeployer deployer;

    @BeforeTest
    public void setUp() throws Exception {
        deployer = new ConfigAdminDeployer();
        deployer.validate();
    }

    @Test
    public void testParsingInvalidSchemaLocation() throws Exception {

        Configadmin parsed = parseConfigurationAdminXmlFile("/invalid-schema-location.xml");

        assertThat(parsed, is( notNullValue() ));
        assertThat(parsed.getConfigurations().size(), is( equalTo( 1 )));
        assertThat(parsed.getFactoryConfigurations().size(), is( equalTo( 0 ) ));

        ConfigurationType cat = parsed.getConfigurations().get(0);
        assertThat(cat.getPid(), is( equalTo("org.ow2.jonas.db.h2.H2DBServiceImpl") ));
        assertThat(cat.getProperties().size(), is( equalTo( 3 ) ));
    }

    @Test
    public void testParsingNoSchemaLocation() throws Exception {

        Configadmin parsed = parseConfigurationAdminXmlFile("/no-schema-location.xml");

        assertThat(parsed, is( notNullValue() ));
        assertThat(parsed.getConfigurations().size(), is( equalTo( 1 )));
        assertThat(parsed.getFactoryConfigurations().size(), is( equalTo( 0 ) ));

        ConfigurationType cat = parsed.getConfigurations().get(0);
        assertThat(cat.getPid(), is( equalTo("org.ow2.jonas.db.h2.H2DBServiceImpl") ));
        assertThat(cat.getProperties().size(), is( equalTo( 3 ) ));
    }

    @Test
    public void testParsingOneFactoryConfiguration() throws Exception {

        Configadmin parsed = parseConfigurationAdminXmlFile("/one-factory-configuration.xml");

        assertThat(parsed, is( notNullValue() ));
        assertThat(parsed.getConfigurations().size(), is( equalTo( 0 )));
        assertThat(parsed.getFactoryConfigurations().size(), is( equalTo( 1 ) ));

        ConfigurationType cat = parsed.getFactoryConfigurations().get(0);
        assertThat(cat.getPid(), is( equalTo("org.ow2.jonas.db.h2.H2DBServiceImpl") ));
        assertThat(cat.getProperties().size(), is( equalTo( 3 ) ));
    }

    @Test
    public void testUndeployRemoveConfigurations() throws Exception {

        // Mocks
        ConfigurationAdmin ca = mock(ConfigurationAdmin.class);
        Configuration configurationOne = mock(Configuration.class);
        Configuration configurationTwo = mock(Configuration.class);
        IFileArchive archive = mock(IFileArchive.class);

        ConfigAdminDeployable deployable = new ConfigAdminDeployable(archive);
        deployable = spy(deployable);

        // Behaviors
        when(deployable.getArchive()).thenReturn(archive);
        when(archive.getURL()).thenReturn(getUrl("/deploy-undeploy-configurations.xml"));
        when(ca.getConfiguration(anyString(), anyString())).thenReturn(configurationOne);
        when(ca.getConfiguration(anyString())).thenReturn(configurationOne);
        when(ca.createFactoryConfiguration(anyString(), anyString())).thenReturn(configurationTwo);

        // Execution
        deployer.bindConfigurationAdmin(ca);
        deployer.deploy(deployable);

        // Assertions (Configurations has been pushed)
        verify(configurationOne).update(any(Dictionary.class));
        verify(configurationTwo).update(any(Dictionary.class));

        deployer.undeploy(deployable);

        // Assertions (Configurations has been deleted)
        verify(configurationOne).delete();
        verify(configurationTwo).delete();

    }

    private Configadmin parseConfigurationAdminXmlFile(String filename) throws Exception {

        // Mocks
        ConfigurationAdmin ca = mock(ConfigurationAdmin.class);
        Configuration configuration = mock(Configuration.class);
        IFileArchive archive = mock(IFileArchive.class);

        ConfigAdminDeployable deployable = new ConfigAdminDeployable(archive);
        deployable = spy(deployable);

        // Behaviors
        when(deployable.getArchive()).thenReturn(archive);
        when(archive.getURL()).thenReturn(getUrl(filename));
        when(ca.getConfiguration(anyString(), anyString())).thenReturn(configuration);
        when(ca.getConfiguration(anyString())).thenReturn(configuration);
        when(ca.createFactoryConfiguration(anyString(), anyString())).thenReturn(configuration);

        // Execution
        deployer.bindConfigurationAdmin(ca);
        deployer.deploy(deployable);

        // Assertions
        verify(configuration).update(any(Dictionary.class));

        return deployable.getAttachedData();
    }



    private URL getUrl(String name) {
        return getClass().getResource(name);
    }
}
