/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: IClusterDaemon.java 21707 2011-09-07 09:44:45Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.cluster.daemon.rest.api;

import org.ow2.jonas.cluster.daemon.api.ClusterDaemonException;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import java.util.List;

/**
 * This class exposes the Cluster Daemon OSGi service as a ReST service.
 * @author Miguel González
 */
@Path("/")
public interface IClusterDaemon {

    @GET
    @Path("/name")
    @Produces(MediaType.TEXT_PLAIN)
    String getObjectName() throws ClusterDaemonException, ClusterDaemonException;

    @POST
    @Path("/name")
    void setObjectName(String name) throws ClusterDaemonException;

    /**
     * @return true if it is an event provider
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/provider/events")
    @Produces(MediaType.TEXT_PLAIN)
    String isEventProvider() throws ClusterDaemonException;

    /**
     * @return true if this managed object implements J2EE State Management
     *         Model
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/manageable/state")
    @Produces(MediaType.TEXT_PLAIN)
    String isStateManageable() throws ClusterDaemonException;

    /**
     * @return true if this managed object implements the J2EE StatisticProvider
     *         Model
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/provider/statistics")
    @Produces(MediaType.TEXT_PLAIN)
    String isStatisticsProvider() throws ClusterDaemonException;

    /**
     * @return list of controlled servers.
     * @throws ClusterDaemonException if an error occurs
     */
    @GET
    @Path("/servers")
    @Produces(MediaType.APPLICATION_XML)
    public List<ServerName> serversNames() throws ClusterDaemonException;

    /**
     * Controlled servers.
     *
     * @return cluster daemon controlled server names.
     * @throws ClusterDaemonException if an error occurs
     */
//    @GET
//    @Path("/servers/controlled")
//    @Produces(MediaType.APPLICATION_XML)
//    public List<ServerName> getControlledServersNames() throws ClusterDaemonException;

    /**
     * @param name JOnAS instance name
     * @return the JAVA_HOME for a specified server name.
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/servers/{name}/javahome")
    @Produces(MediaType.TEXT_PLAIN)
    String getJavaHome4Server(@PathParam("name") String name)
            throws ClusterDaemonException;

    /**
     * get <code>JONAS_ROOT</code> for the given server.
     *
     * @param name JOnAS instance name
     * @return the JONAS_ROOT for a specified server name
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/servers/{name}/jonas/root")
    @Produces(MediaType.TEXT_PLAIN)
    String getJonasRoot4Server(@PathParam("name") String name)
            throws ClusterDaemonException;

    /**
     * Get <code>JONAS_BASE</code> for the given server.
     *
     * @param name JOnAS instance name
     * @return the JONAS_BASE for a specified server name.
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/servers/{name}/jonas/base")
    @Produces(MediaType.TEXT_PLAIN)
    String getJonasBase4Server(@PathParam("name") String name)
            throws ClusterDaemonException;

    /**
     * Get command for the given server.
     *
     * @param name JOnAS instance name
     * @return the user command for a specified server name
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/servers/{name}/jonas/cmd")
    @Produces(MediaType.TEXT_PLAIN)
    String getJonasCmd4Server(@PathParam("name")String name)
            throws ClusterDaemonException;

    /**
     * Get extra parameters for the given server.
     *
     * @param name JOnAS instance name
     * @return xprm for a specified server name
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/servers/{name}/extra")
    @Produces(MediaType.TEXT_PLAIN)
    String getXprm4Server(@PathParam("name") String name)
            throws ClusterDaemonException;

    /**
     * True if <code>autoboot</code> is enabled for the given server.
     *
     * @param name JOnAS instance name
     * @return autoBoot value for a specified server name
     * @throws ClusterDaemonException if an error occurs
     */
    @GET
    @Path("/servers/{name}/autoboot")
    @Produces(MediaType.TEXT_PLAIN)
    String getAutoBoot4Server(@PathParam("name") String name) throws ClusterDaemonException;

    /**
     * Reload the configuration.
     *
     * @throws ClusterDaemonException if an error occurs
     */
    @POST
    @Path("/config")
    void reloadConfiguration() throws ClusterDaemonException;

    /**
     * Add a server configuration.
     *
     * @param name        server name
     * @param description server description
     * @param javaHome    JAVA_HOME dir
     * @param jonasBase   JONAS_BASE dir
     * @param jonasRoot   JONAS_ROOT dir
     * @param xprm        extra JVM parameters
     * @param autoBoot    automatic start
     * @param jonasCmd    user command
     * @param saveIt      save its configuration
     * @throws ClusterDaemonException if an error occurs
     */
    @POST
    @Path("/servers/{name}")
    public void addServer(@PathParam("name") String name,
                   @FormParam("description") String description,
                   @FormParam("jonasRoot") String jonasRoot,
                   @FormParam("jonasBase") String jonasBase,
                   @FormParam("javaHome")String javaHome,
                   @FormParam("xprm") String xprm,
                   @FormParam("autoboot") String autoBoot,
                   @FormParam("jonasCmd") String jonasCmd,
                   @FormParam("saveit") String saveIt)
            throws ClusterDaemonException;

    /**
     * Modify a server configuration.
     *
     * @param name        server name
     * @param description server description
     * @param javaHome    JAVA_HOME dir
     * @param jonasBase   JONAS_BASE dir
     * @param jonasRoot   JONAS_ROOT dir
     * @param xprm        extra JVM parameters
     * @param autoBoot    automatic start
     * @param jonasCmd    user command
     * @throws ClusterDaemonException if an error occurs
     */
    @PUT
    @Path("/servers/{name}")
    public void modifyServer(@PathParam("name") String name,
                      @FormParam("description") String description,
                      @FormParam("javaHome") String javaHome,
                      @FormParam("jonasRoot") String jonasRoot,
                      @FormParam("jonasBase") String jonasBase,
                      @FormParam("xprm") String xprm,
                      @FormParam("autoboot") String autoBoot,
                      @FormParam("jonasCmd") String jonasCmd)
            throws ClusterDaemonException;

    /**
     * Remove this server from cluster daemon control.
     *
     * @param serverName the server to remove
     * @param saveIt     if true then flush the configuration
     * @throws ClusterDaemonException if an error occurs
     */
    @DELETE
    @Path("/servers/{name}")
    public void removeServer(@PathParam("name") String serverName,
                             @DefaultValue("true") @QueryParam("save")
                             String saveIt)
            throws ClusterDaemonException;

    /**
     * Save the configuration.
     *
     * @throws ClusterDaemonException if an error occurs
     */
    @POST
    @Path("/config/save")
    public void saveConfiguration() throws ClusterDaemonException;

    /**
     * Start a JOnAS instance.
     *
     * @param name       instance name
     * @param domainName domain name
     * @param prm        extra parameters
     * @throws ClusterDaemonException if an error occurs
     */
    @POST
    @Path("/servers/{name}/domain/{domain}/start")
    @Consumes(MediaType.TEXT_PLAIN)
    void startJOnAS(String prm, @PathParam("name")String name,
                    @PathParam("domain") String domainName)
            throws ClusterDaemonException;

    /**
     * Stop a JOnAS instance.
     *
     * @param name instance name
     * @throws ClusterDaemonException if an error occurs
     */
    @POST
    @Path("/servers/{name}/stop")
    void stopJOnAS(@PathParam("name") String name) throws ClusterDaemonException;

    /**
     * Halt a JOnAS instance.
     *
     * @param name instance name
     * @throws ClusterDaemonException if an error occurs
     */
    @POST
    @Path("/servers/{name}/halt")
    void haltJOnAS(@PathParam("name") String name) throws ClusterDaemonException;

    /**
     * Ping a JOnAS instance (MBean interface).
     *
     * @param name instance name
     * @return exit code of the ping (0 ok, 1 ko)
     * @throws ClusterDaemonException if an error occurs
     */
    @GET
    @Path("/servers/{name}/ping")
    @Produces(MediaType.TEXT_PLAIN)
    String pingJOnAS(@PathParam("name") String name) throws ClusterDaemonException;

    /**
     * Start all the JOnAS instances configured with auto-reboot.
     *
     * @param domainName domain name
     * @param prm        extra parameters
     * @return the nodes list with an indicator started/starting failed
     * @throws ClusterDaemonException if an error occurs
     */
    @POST
    @Path("/servers/domain/{domain}/start")
    @Produces(MediaType.TEXT_PLAIN)
    @Consumes(MediaType.TEXT_PLAIN)
    String startAllJOnAS(String prm,
                         @PathParam("domain") String domainName)
            throws ClusterDaemonException;

    /**
     * Stop all the JOnAS instances.
     *
     * @return the nodes list with an indicator stopped/stopping failed
     * @throws ClusterDaemonException any.
     */
    @POST
    @Path("/servers/stop")
    String stopAllJOnAS() throws ClusterDaemonException;

    /**
     * Stop the cluster daemon instance.
     *
     * @throws ClusterDaemonException any.
     */
    @POST
    @Path("/stop")
    void stopClusterDaemon() throws ClusterDaemonException;

    /**
     * Get jmx url of the cluster daemon.
     *
     * @return clusterd jmx url
     * @throws ClusterDaemonException if an error occurs
     */
    @GET
    @Path("/jmx")
    @Produces(MediaType.TEXT_PLAIN)
    String getJmxUrl() throws ClusterDaemonException;

    /**
     * Get available processors of the OS.
     *
     * @return Operating system processors number
     * @throws ClusterDaemonException if an error occurs
     */
    @GET
    @Path("/os/processors")
    @Produces(MediaType.TEXT_PLAIN)
    public String getOperatingSystemAvailableProcessors() throws ClusterDaemonException;

    /**
     * Get OS name.
     *
     * @return OS name
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/os/name")
    @Produces(MediaType.TEXT_PLAIN)
    public String getOperatingSystemName() throws ClusterDaemonException;

    /**
     * get OS architecture.
     *
     * @return Os architecture
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/os/arch")
    @Produces(MediaType.TEXT_PLAIN)
    public String getOperatingSystemArch() throws ClusterDaemonException;

    /**
     * Get OS version.
     *
     * @return OS version
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/os/version")
    @Produces(MediaType.TEXT_PLAIN)
    public String getOperatingSystemVersion() throws ClusterDaemonException;

    /**
     * Get spec. vendor.
     *
     * @return Spec vendor
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/spec/vendor")
    @Produces(MediaType.TEXT_PLAIN)
    public String getRunTimeSpecVendor() throws ClusterDaemonException;

    /**
     * Get runtime spec. version.
     *
     * @return Spec version
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/spec/version")
    @Produces(MediaType.TEXT_PLAIN)
    public String getRunTimeSpecVersion() throws ClusterDaemonException;

    /**
     * Get runtime vm name.
     *
     * @return the Vm name
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/vm/name")
    @Produces(MediaType.TEXT_PLAIN)
    public String getRunTimeVmName() throws ClusterDaemonException;

    /**
     * Get runtime vendor.
     *
     * @return Vm vendor
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/vm/vendor")
    @Produces(MediaType.TEXT_PLAIN)
    public String getRunTimeVmVendor() throws ClusterDaemonException;

    /**
     * Get runtime vm version.
     *
     * @return Vm version
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/vm/version")
    @Produces(MediaType.TEXT_PLAIN)
    public String getRunTimeVmVersion() throws ClusterDaemonException;

    /**
     * Get dynamic host attributes.
     *
     * @return hashtable with dynamic attributes keys and values.
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/attributes/dynamic")
    @Produces(MediaType.APPLICATION_XML)
    public List<Attribute> getDynamicHostAttributes() throws ClusterDaemonException;

    /**
     * Retrieve the state of the given controlled server.
     *
     * @param srvName server's name.
     * @return true if the server is running
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/servers/{name}/state")
    @Produces(MediaType.TEXT_PLAIN)
    public String checkServerState(@PathParam("name") String srvName)
            throws ClusterDaemonException;

    /**
     * Get remote Vm used Memory.
     *
     * @return the value of current used memory
     * @throws ClusterDaemonException any.
     */

    @GET
    @Path("/runtime/vm/memory/current")
    @Produces(MediaType.TEXT_PLAIN)
    public String getVmCurrentUsedMemory() throws ClusterDaemonException;

    /**
     * Getting remote Vm Total Memory.
     *
     * @return the value of Vm Total memory
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/vm/memory")
    @Produces(MediaType.TEXT_PLAIN)
    public String getVmTotalMemory() throws ClusterDaemonException;

    /**
     * Getting remote Vm's Current used Heap memory.
     *
     * @return the value of Vm's Current used Heap memory
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/vm/memory/heap")
    @Produces(MediaType.TEXT_PLAIN)
    public String getVmCurrentUsedHeapMemory() throws ClusterDaemonException;

    /**
     * Getting remote Vm's Current used non Heap memory.
     *
     * @return the value of Vm's Current used non Heap memory
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/runtime/vm/memory/nonheap")
    @Produces(MediaType.TEXT_PLAIN)
    public String getVmCurrentUsedNonHeapMemory() throws ClusterDaemonException;

    /**
     * Getting Operating system Current used space.
     *
     * @return the value of Operating system Current used space
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/os/space/used")
    @Produces(MediaType.TEXT_PLAIN)
    public String getOsCurrentUsedSpace() throws ClusterDaemonException;

    /**
     * Getting Operating system Current used space.
     *
     * @return the value of Operating system Total space
     * @throws ClusterDaemonException any.
     */
    @GET
    @Path("/os/space")
    @Produces(MediaType.TEXT_PLAIN)
    public String getOsTotalSpace() throws ClusterDaemonException;

}
