/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2004-2011 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Jms.java 21593 2011-08-08 12:29:07Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.antmodular.jonasbase.jms;

import org.ow2.jonas.antmodular.jonasbase.bootstrap.AbstractJOnASBaseAntTask;
import org.ow2.jonas.antmodular.jonasbase.bootstrap.JReplace;

import java.util.StringTokenizer;


/**
 * Allow to configure the JMS service.
 * @author Florent Benoit
 */
public class Jms extends AbstractJOnASBaseAntTask {

    /**
     * Info for the logger.
     */
    private static final String INFO = "[JMS] ";

    /**
     * Default host name.
     */
    private static final String DEFAULT_HOST = "localhost";

    /**
     * Default port number.
     */
    private static final String DEFAULT_PORT = "16010";

    /**
     * Token for the end of the joramAdmin configuration file.
     */
    private static final String TOKEN_END_CONF_FILE = "</JoramAdmin>";

    /**
     * Name of Joram configuration file.
     */
    public static final String JORAM_CONF_FILE = "a3servers.xml";

    /**
     * Name of Joram admin configuration file (resource adaptor).
     */
    public static final String JORAM_ADMIN_CONF_FILE = "joramAdmin.xml";

    /**
     * Default constructor.
     */
    public Jms() {
        super();
    }

    /**
     * Set the host name for Joram.
     * @param host host name for Joram
     */
    public void setHost(final String host) {

        // For JMS
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JORAM_CONF_FILE);
        propertyReplace.setToken(DEFAULT_HOST);
        propertyReplace.setValue(host);
        propertyReplace.setLogInfo(INFO + "Setting Joram host name to : " + host + " in "
                + JORAM_CONF_FILE + " file.");
        addTask(propertyReplace);

        // for RAR file
        propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JORAM_ADMIN_CONF_FILE);
        propertyReplace.setToken(DEFAULT_HOST);
        propertyReplace.setValue(host);
        propertyReplace.setLogInfo(INFO + "Setting Joram host name to : " + host + " in "
                + JORAM_ADMIN_CONF_FILE + " file.");
        addTask(propertyReplace);

        // Patch the RAR file
        JmsRa jmsRa = new JmsRa();
        jmsRa.setServerHost(host);
        addTask(jmsRa);

    }

    /**
     * Set the port number for Joram.
     * @param portNumber the port for Joram
     */
    public void setPort(final String portNumber) {

        // For JMS
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JORAM_CONF_FILE);
        propertyReplace.setToken(DEFAULT_PORT);
        propertyReplace.setValue(portNumber);
        propertyReplace.setLogInfo(INFO + "Setting Joram port number to : " + portNumber + " in "
                + JORAM_CONF_FILE + " file.");
        addTask(propertyReplace);

        // for RAR file
        propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JORAM_ADMIN_CONF_FILE);
        propertyReplace.setToken(DEFAULT_PORT);
        propertyReplace.setValue(portNumber);
        propertyReplace.setLogInfo(INFO + "Setting Joram port number to : " + portNumber + " in "
                + JORAM_ADMIN_CONF_FILE + " file.");
        addTask(propertyReplace);

        // Patch the RAR file
        JmsRa jmsRa = new JmsRa();
        jmsRa.setServerPort(portNumber);
        addTask(jmsRa);

    }

    /**
     * Set the initial topics when JOnAS start.
     * @param initialTopics comma separated list of topics
     */
    public void setInitialTopics(final String initialTopics) {
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JORAM_ADMIN_CONF_FILE);
        propertyReplace.setToken(TOKEN_END_CONF_FILE);
        String tokenValue = "";
        StringTokenizer st = new StringTokenizer(initialTopics, ",");

        while (st.hasMoreTokens()) {
            String topic = st.nextToken();
            tokenValue += " <Topic name=\""
                          + topic + "\">"
                          + "\n"
                          + "  <freeReader/>"
                          + "\n"
                          + "  <freeWriter/>"
                          + "\n"
                          + "  <jndi name=\""
                          + topic
                          + "\"/>"
                          + "\n"
                          + " </Topic>"
                          + "\n";
        }

        tokenValue += TOKEN_END_CONF_FILE;

        propertyReplace.setValue(tokenValue);
        propertyReplace.setLogInfo(INFO + "Setting initial topics to : " + initialTopics);
        addTask(propertyReplace);
    }

    /**
     * Set the initial queues when JOnAS start.
     * @param initialQueues comma separated list of topics
     */
    public void setInitialQueues(final String initialQueues) {
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JORAM_ADMIN_CONF_FILE);
        propertyReplace.setToken(TOKEN_END_CONF_FILE);

        String tokenValue = "";
        StringTokenizer st = new StringTokenizer(initialQueues, ",");

        while (st.hasMoreTokens()) {
            String queue = st.nextToken();
            tokenValue += " <Queue name=\""
                          + queue
                          + "\">"
                          + "\n"
                          + "  <freeReader/>"
                          + "\n"
                          + "  <freeWriter/>"
                          + "\n"
                          + "  <jndi name=\"" + queue + "\"/>"
                          + "\n"
                          + " </Queue>"
                          + "\n";
        }

        tokenValue += TOKEN_END_CONF_FILE;

        propertyReplace.setValue(tokenValue);
        propertyReplace.setLogInfo(INFO + "Setting initial queues to : " + initialQueues);
        addTask(propertyReplace);
    }

    /**
     * Execute all task
     */
    public void execute() {
        super.execute();
        super.executeAllTask();
    }
}
