/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: AddonDeployerComponent.java 21827 2011-10-23 22:48:21Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.addon.deploy.impl.deployer.service;

import org.osgi.framework.BundleContext;
import org.ow2.jonas.addon.deploy.api.deployer.IAddonDeployer;
import org.ow2.jonas.addon.deploy.impl.deployer.AddonDeployerImpl;
import org.ow2.jonas.addon.deploy.impl.util.AddonDeployerLog;
import org.ow2.jonas.addon.deploy.impl.util.AddonUtil;
import org.ow2.jonas.configuration.ConfigurationManager;
import org.ow2.jonas.lib.service.AbsServiceImpl;
import org.ow2.jonas.lib.work.DeployerLogException;
import org.ow2.jonas.management.ServiceManager;
import org.ow2.jonas.properties.ServerProperties;
import org.ow2.jonas.service.ServiceException;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManager;
import org.ow2.util.ee.deploy.api.helper.IDeployableHelper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.io.File;
import java.io.IOException;

/**
 * Service that will register an AddonDeployerImpl
 * @author Jeremy Cazaux
 */
public class AddonDeployerComponent extends AbsServiceImpl {

    /**
     * Deployer Manager. (that allow us to deploy Deployable).
     */
    private IDeployerManager deployerManager;

    /**
     * The Addon Deployer
     */
    private IAddonDeployer addonDeployer;

    /**
     * The DeployableHelper  which analyze an archive and build the associated Deployable object
     */
    private IDeployableHelper deployableHelper;

    /**
     * Server properties
     */
    private ServerProperties serverProperties;

    /**
     * The logger
     */
    private static Log logger = LogFactory.getLog(AddonDeployerComponent.class);

    /**
     * The context of the component
     */
    private BundleContext bundleContext;

    /**
     * The configuration manager
     */
    private ConfigurationManager configurationManager;

    /**
     * Default constructor
     * @param bundleContext The context
     */
    public AddonDeployerComponent(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    /**
     * Start the service
     * @throws org.ow2.jonas.service.ServiceException
     */
    @Override
    protected void doStart() throws ServiceException {

        //ceate parent directory of the log file, if !exist
        File logFile = new File(AddonUtil.getAddonLogFile(this.serverProperties));
        if (!logFile.getParentFile().exists()) {
            logFile.getParentFile().mkdirs();
        }

        //create the log file
        try {
            logFile.createNewFile();
        } catch (IOException e) {
            logger.error("Cannot create the log file for addon deployer");
        }

        //create a new log for the Addon deployer
        AddonDeployerLog deployerLog = null;
        try {
            deployerLog = new AddonDeployerLog(logFile);
        } catch (DeployerLogException e) {
            logger.error("Cannot create a DeployerLog for Addon");
        }

        //Create a new AddonDeployer
        this.addonDeployer = new AddonDeployerImpl(this.serverProperties, this.deployableHelper, deployerLog,
                this.deployerManager, this.bundleContext, this.configurationManager);

        //register the addon deployer
        this.deployerManager.register(this.addonDeployer);
    }

    /**
     * Stop the service
     * @throws org.ow2.jonas.service.ServiceException
     */
    @Override
    protected void doStop() throws ServiceException {
        this.deployerManager.unregister(this.addonDeployer);
    }

    /**
     * @param deployerManager the {@link org.ow2.util.ee.deploy.api.deployer.IDeployerManager} to set
     */
    public void setDeployerManager(final IDeployerManager deployerManager) {
        this.deployerManager = deployerManager;
    }

    /**
     * @param serverProperties ServerProperties to set
     */
    public void setServerProperties(final ServerProperties serverProperties) {
        this.serverProperties = serverProperties;
    }

    /**
     * @param deployableHelper The DeployableHelper which analyze an archive and build the associated Deployable object
     */
    public void setDeployableHelper(final IDeployableHelper deployableHelper) {
        this.deployableHelper = deployableHelper;
    }

    /**
     * @param configurationManager The configuration manager to registered
     */
    public void registerConfigurationManager(ConfigurationManager configurationManager) {
        this.configurationManager = configurationManager;
    }

    /**
     * Unregistered the configuration manager
     */
    public void unregisterConfigurationManager() {
        this.configurationManager = null;
    }
}
