/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: AddonLogEntry.java 21768 2011-09-28 19:19:30Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.addon.deploy.impl.util;

import org.ow2.jonas.addon.deploy.api.util.IAddonLogEntry;
import org.ow2.jonas.lib.work.LogEntryImpl;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

/**
 * Class wich represent an entry in an addon log file
 */
public class AddonLogEntry extends LogEntryImpl implements IAddonLogEntry {

    /**
     * The name of the addon
     */
    private String name;

    /**
     * The list of OSGi deployables
     */
    private List<String> osgiDeployables;

    /**
     * Open tag for the list of OSGi deployables
     */
    public final String OSGI_OPEN_TAG = "osgi{";

    /**
     * Separator between the list of OSGi deployables
     */
    public final String OSGI_SEPARATOR = ",";

    /**
     * End tag for the list of OSGi deployables
     */
    public final String OSGI_END_TAG = "}";

    /**
     * The logger
     */
    private static Log logger = LogFactory.getLog(AddonLogEntry.class);

    /**
     * Constructor of a log entry.
     * @param name the name of the addon
     * @param original the file to copy
     * @param copy     name of the copy
     */
    public AddonLogEntry(final String name, final File original, final File copy) {
        super(original, copy);
        this.name = name;
        this.osgiDeployables = new ArrayList<String>();
    }

    /**
     * @return the name of the addon
     */
    public String getName() {
        return name;
    }

    /**
     * @return the OSGi deployables as a String with the following structure: osgi{path1,path2,path3}
     */
    public String getOSGiDeployablesAsAString() {
        String result = OSGI_OPEN_TAG;
        int i = 0;
        while (i < this.osgiDeployables.size() - 1) {
            result += this.osgiDeployables.get(i) + OSGI_SEPARATOR;
            i++;
        }

        if (i <this.osgiDeployables.size()) {
            result += this.osgiDeployables.get(i);
        }

        result += OSGI_END_TAG;

        return result;
    }

    /**
     * @return the OSGi deployables
     */
    public List<String> getOSGiDeployables() {
        return this.osgiDeployables;
    }

    /**
     * Add an OSGi deployable to the list of OSGi deployables
     * @param osgiDeployable The path to an OSGi deployable
     */
    public void addOSGiDeployable(final String osgiDeployable) {
        this.osgiDeployables.add(osgiDeployable);
    }

    /**
     * @param osgiDeployables List of OSGi deployables to set
     */
    public void setOSGiDeployables(final List<String> osgiDeployables) {
        this.osgiDeployables = osgiDeployables;
    }

    /**
     * @param osgiDeployables The well formed element which indicate the list of OSGiDeployable with the following
     * structure: osgi{path1,path2,path3}
     */
    public void setOSGiDeployables(final String osgiDeployables) {
        this.osgiDeployables = new ArrayList<String>();

        if (osgiDeployables.startsWith(OSGI_OPEN_TAG) && osgiDeployables.endsWith(OSGI_END_TAG)) {
            String elements = osgiDeployables.substring(OSGI_OPEN_TAG.length(), osgiDeployables.length() - OSGI_END_TAG.length());
            StringTokenizer stringTokenizer = new StringTokenizer(elements, OSGI_SEPARATOR);
            while (stringTokenizer.hasMoreElements()) {
                this.osgiDeployables.add(stringTokenizer.nextElement().toString());
            }
        }
    }
}
