/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JndiTest.java 22065 2012-01-31 13:14:39Z joaninh $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.itests.cargo.jrmp;

import org.ow2.easybeans.osgi.tests.AbsTesting;
import org.ow2.jonas.tests.applications.jndi.ejb3.Jndi;
import org.testng.Assert;
import org.testng.annotations.AfterSuite;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import java.util.Hashtable;

/**
 * Integration tests of the JNDI service
 * @author Jeremy Cazaux
 */
public class JndiTest extends AbsTesting  {

    /**
     *  The context name of the EJB to lookup
     */
    public final static String JNDI_EJB_KEY = "ejb/test/jndi";

    /**
     * Key of first entry of the sample example
     */
    public final static String SAMPLE_EXAMPLE_JNDI_KEY_1 = "jndi/developer";

    /**
     * Value of the first entry of the sample example
     */
    public final static String SAMPLE_EXAMPLE_JNDI_VALUE_1 = "The JOnAS Team";

    /**
     * Type of the first entry of the sample example
     */
    public final static String SAMPLE_EXAMPLE_JNDI_TYPE_1 ="java.lang.String";

    /**
     * Key of second entry of the sample example
     */
    public final static String SAMPLE_EXAMPLE_JNDI_KEY_2 = "jndi/arch";

    /**
     * Value of the second entry of the sample example
     */
    public final static String SAMPLE_EXAMPLE_JNDI_VALUE_2 = "64";

    /**
     * Type of the second entry of the sample example
     */
    public final static String SAMPLE_EXAMPLE_JNDI_TYPE_2 = "java.lang.Integer";

    /**
     * JNDI key of the empty value example
     */
    public final static String EMPTY_VALUE_EXAMPLE_JNDI_KEY = "test/value/empty";

    /**
     * The system property for the port of the smartclient
     */
    public final static String SMART_CLIENT_PORT_SYSTEM_PROPERTY = "smartclient.port";

    /**
     * Default InitialContextFactory to use.
     */
    private static final String INITIAL_CONTEXT_FACTORY = "org.ow2.easybeans.component.smartclient.spi.SmartContextFactory";

    /**
     * URL of the provider (without the port)
     */
    private static final String PROVIDER_URL = "smart://localhost:";

    /**
     * The context
     */
    private Context initialContext;

    /*
     * The remote EJB
     */
    private Jndi ejb;

    /**
     * The old classloader
     */
    private ClassLoader oldClassLoader;

    /**
     * Initialize the context
     */
    @BeforeClass
    public void initInitialContext() throws Exception {
        super.init();

        this.initialContext = getInitialContext();

        //lookup the EJB from the JNDI context
        this.ejb = null;
        try {
            this.ejb = Jndi.class.cast(this.initialContext.lookup(JNDI_EJB_KEY));
        } catch (NamingException e) {
            Assert.fail("Cannot lookup context " + JNDI_EJB_KEY, e);
        }

        //nedd to fix a classloader IRMI issue with the smartclient (some class wasn't downloaded...). Ugly but usefull
        //to fix a IRMI issue
        this.oldClassLoader = Thread.currentThread().getContextClassLoader();
        ClassLoader ejbClassLoader = this.ejb.getClass().getClassLoader();
        Thread.currentThread().setContextClassLoader(ejbClassLoader);
    }

    /**
     * JNDI injection test of a sample entry
     */
    @Test
    public void testThatSampleExampleIsOk() throws Exception {
        //test the first entry
        Object val1 = this.ejb.getValue(SAMPLE_EXAMPLE_JNDI_KEY_1);
        Assert.assertEquals(String.valueOf(val1), SAMPLE_EXAMPLE_JNDI_VALUE_1);
        Assert.assertEquals(val1.getClass().getName(),SAMPLE_EXAMPLE_JNDI_TYPE_1);

        //test the second entry
        Object val2 = this.ejb.getValue(SAMPLE_EXAMPLE_JNDI_KEY_2);
        Assert.assertEquals(String.valueOf(val2), SAMPLE_EXAMPLE_JNDI_VALUE_2);
        Assert.assertEquals(val2.getClass().getName(),SAMPLE_EXAMPLE_JNDI_TYPE_2);
    }

    /**
     * Tetst that empty value is Ok
     */
    @Test
    public void testThatEmptyValueIsOk() {
        Object value = this.ejb.getValue(EMPTY_VALUE_EXAMPLE_JNDI_KEY);
        Assert.assertTrue(value != null);
    }

    /**
     * @return Returns the InitialContext.
     * @throws javax.naming.NamingException If the Context cannot be created.
     */
    private static Context getInitialContext() throws NamingException {
        Hashtable<String, Object> env = new Hashtable<String, Object>();
        env.put(Context.INITIAL_CONTEXT_FACTORY, INITIAL_CONTEXT_FACTORY);
        env.put(Context.PROVIDER_URL, PROVIDER_URL + System.getProperty(SMART_CLIENT_PORT_SYSTEM_PROPERTY));

        return  new InitialContext(env);
    }

    /**
     * Reset the classloader after that all test methods in the current class have been run.
     */
    @AfterSuite
    public void resetClassLoader() {
        Thread.currentThread().setContextClassLoader(this.oldClassLoader);
    }
}
