/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: AddonMetaData.java 22060 2012-01-29 12:43:56Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.addon.deploy.impl.deployer;

import org.ow2.jonas.addon.deploy.impl.util.Range;
import org.ow2.jonas.addon.properties.ServicePropertiesImpl;
import org.ow2.jonas.properties.ServiceProperties;
import java.io.File;
import java.util.Properties;

/**
 * Represents metadata of an Addon
 */
public class AddonMetaData {

    /**
     * The name of this addon
     */
    private String name;

    /**
     * The description of this addon
     */
    private String description;

    /**
     * The author of this addon
     */
    private String author;

    /**
     * The licence of this addon
     */
    private String licence;

    /**
     * Indicates a version of JOnAS that this addon can run in.
     */
    private Range jonasVersions;

    /**
     * The autostart property
     */
    private Boolean autostart;

    /**
     * Indicates a list of version of the jvm that this plugin can run in.
     */
    private Range jvmVersions;

    /**
     * Provides property for the resolver
     */
    private String provides;

    /**
     * Requirements property for the resolver
     */
    private String requirements;

    /**
     * Properties of the service (if the addon represents a JOnAS service)
     */
    private ServiceProperties serviceProperties;

    /**
     * The metadata file
     */
    private File addonMetaDataFile;

    /**
     * The name of the JOnAS service associated to the addon. Null if the addon is not a JOnAS service
     */
    private String service;

    /**
     * The name of the implementation of the JOnAS service
     */
    private String implementation;

    /**
     * Default constructor
     */
    public AddonMetaData() {
        this.autostart = false;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @return the description
     */
    public String getDescription() {
        return description;
    }

    /**
     * @return the author
     */
    public String getAuthor() {
        return author;
    }

    /**
     * @return the licence
     */
    public String getLicence() {
        return licence;
    }

    /**
     * Check if the given jonas version is support
     * @param jonasVersion  The jonas version to check
     */
    public boolean isJOnASVersionSupported(final String jonasVersion) {
        return this.jonasVersions.isSupported(jonasVersion);
    }

    /**
     * @return the autostart property
     */
    public Boolean getAutostart() {
        return autostart;
    }

    /**
     * Check if the given jvm version is support
     * @param jvmVersion  The jvm version to check
     */
    public boolean isJvmVersionSupported(final String jvmVersion) {
        return this.jvmVersions.isSupported(jvmVersion);
    }

    /**
     * @return provides properties for the resolver
     */
    public String getProvides() {
        return provides;
    }

    /**
     * @return requirements properties
     */
    public String getRequirements() {
        return requirements;
    }

    /**
     * @param name The name of this addon to set
     */
    public void setName(final String name) {
        this.name = name;
    }

    /**
     * @param description the description to set
     */
    public void setDescription(final String description) {
        this.description = description;
    }

    /**
     * @param author The author to set
     */
    public void setAuthor(final String author) {
        this.author = author;
    }

    /**
     * @param licence The licence to set
     */
    public void setLicence(final String licence) {
        this.licence = licence;
    }

    /**
     * @param jonasVersions List of jonas versions compatible with this addon  to set
     */
    public void setJonasVersions(final Range jonasVersions) {
        this.jonasVersions = jonasVersions;
    }

    /*
     * @param autostart Autostart properties to set
     */
    public void setAutostart(final Boolean autostart) {
        this.autostart = autostart;
    }

    /**
     * @param jvmVersions A list of Jvm version to set
     */
    public void setJvmVersions(final Range jvmVersions) {
        this.jvmVersions = jvmVersions;
    }

    /**
     * @param provides Provides properties to set
     */
    public void setProvides(final String provides) {
        this.provides = provides;
    }

    /**
     * @param requirements Requirements properties to set
     */
    public void setRequirements(final String requirements) {
        this.requirements = requirements;
    }

    /*
     * @param properties Properties of the addon to set
     */
    public void setServiceProperties(final Properties properties) {
        this.serviceProperties = new ServicePropertiesImpl(properties, this.service);
    }

    /**
     * @return properties of the addon
     */
    public ServiceProperties getServiceProperties() {
        return this.serviceProperties;
    }

    /**
     * @return the metadata file
     */
    public File getAddonMetaDataFile() {
        return this.addonMetaDataFile;
    }

    /**
     * @param addonMetaDataFile The metadata file to set
     */
    public void setAddonMetaDataFile(final File addonMetaDataFile) {
        this.addonMetaDataFile = addonMetaDataFile;
    }

    /**
     * @return the name of the JOnAS service associated to the addon. Null if the addon is not a JOnAS service
     */
    public String getService() {
        return service;
    }

    /**
     * @param service the name of the JOnAS service associated to the addon to set
     */
    public void setService(final String service) {
        this.service = service;
    }

    /**
     * @return the name of the implementation of the JOnAS service
     */
    public String getImplementation() {
        return implementation;
    }

    /**
     * @param implementation the name of the implementation of the JOnAS service to set
     */
    public void setImplementation(final String implementation) {
        this.implementation = implementation;
    }

    /**
     * @return true if the addon represents a JOnAS service
     */
    public boolean isJOnASService() {
        return this.service != null;
    }
}
