/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: OSGIContext.java 21566 2011-08-08 12:28:12Z cazauxj $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.web.tomcat6.osgi.httpservice;

import javax.servlet.ServletContext;

import org.osgi.framework.Bundle;
import org.osgi.service.http.HttpContext;


import org.ow2.jonas.web.tomcat6.JOnASStandardContext;
import org.ow2.jonas.web.tomcat6.osgi.httpservice.OSGIWrapper;



/**
 * {@link Context} implementation for holding OSGI servlet wrappers. Only the
 * {@link #getServletContext()} method is overridden.
 * @author Guillaume Porcher
 */
public class OSGIContext extends JOnASStandardContext {

    /**
     * Serial version UID.
     */
    private static final long serialVersionUID = -8093637123939061548L;

    /**
     * The {@link HttpContext} used during servlet or resource registration.
     */
    private final HttpContext httpContext;

    /**
     * Default constructor.
     * @param context The HttpContext used during servlet or resource
     *        registration.
     * @param callerBundle The bundle that uses the HttpService.
     */
    public OSGIContext(final HttpContext context) {
        this.httpContext = context;
    }

    /**
     * Return the servlet context for which this Context is a facade. The
     * returned context is an instance of {@link OSGIServletContext}.
     * @return the servlet context for which this Context is a facade.
     */
    @Override
    public ServletContext getServletContext() {
        // Initialize if not yet done
        if (context == null) {
            this.context = new OSGIServletContext(this, this.httpContext);
        }
        return context;
    }

    /**
     * @return The httpContext used during servlet or resource registration.
     */
    public HttpContext getHttpContext() {
        return this.httpContext;
    }

    /**
     * Add a new servlet mapping, replacing any existing mapping for the
     * specified pattern. In OSGIContext the OSGiWrapper added are forced to
     * update their mapping.
     * @param pattern URL pattern to be mapped
     * @param name Name of the corresponding servlet to execute
     * @param jspWildCard true if name identifies the JspServlet and pattern
     *        contains a wildcard; false otherwise
     * @exception IllegalArgumentException if the specified servlet name is not
     *            known to this Context
     */
    @Override
    public void addServletMapping(final String pattern, final String name, final boolean jspWildCard) throws IllegalArgumentException {
        super.addServletMapping(pattern, name, jspWildCard);
        ((OSGIWrapper) findChild(name)).updateMapping();
    }

}
