/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: AddonDeployerComponent.java 21987 2011-12-12 20:57:22Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.addon.deploy.impl.deployer.service;

import org.osgi.framework.BundleContext;
import org.ow2.jonas.addon.deploy.api.deployer.IAddonDeployer;
import org.ow2.jonas.addon.deploy.api.deployer.IAddonDeployerLog;
import org.ow2.jonas.addon.deploy.impl.cleaner.AddonCleanTask;
import org.ow2.jonas.addon.deploy.impl.deployer.AddonDeployerImpl;
import org.ow2.jonas.addon.deploy.impl.util.AddonDeployerLog;
import org.ow2.jonas.addon.deploy.impl.util.AddonUtil;
import org.ow2.jonas.lib.service.AbsServiceImpl;
import org.ow2.jonas.lib.work.DeployerLog;
import org.ow2.jonas.management.ServiceManager;
import org.ow2.jonas.workcleaner.CleanTask;
import org.ow2.jonas.workcleaner.DeployerLogException;
import org.ow2.jonas.properties.ServerProperties;
import org.ow2.jonas.service.ServiceException;
import org.ow2.jonas.workcleaner.IDeployerLog;
import org.ow2.jonas.workcleaner.WorkCleanerService;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManager;
import org.ow2.util.ee.deploy.api.helper.IDeployableHelper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.io.File;
import java.io.IOException;

/**
 * Service that will register an AddonDeployerImpl
 * @author Jeremy Cazaux
 */
public class AddonDeployerComponent extends AbsServiceImpl {

    /**
     * Deployer Manager. (that allow us to deploy Deployable).
     */
    private IDeployerManager deployerManager;

    /**
     * The Addon Deployer
     */
    private IAddonDeployer addonDeployer;

    /**
     * The DeployableHelper  which analyze an archive and build the associated Deployable object
     */
    private IDeployableHelper deployableHelper;

    /**
     * Server properties
     */
    private ServerProperties serverProperties;

    /**
     * The logger
     */
    private static Log logger = LogFactory.getLog(AddonDeployerComponent.class);

    /**
     * The context of the component
     */
    private BundleContext bundleContext;

    /**
     * The service manager
     */
    private ServiceManager serviceManager;

    /**
     * The addon deployer log
     */
    private IAddonDeployerLog deployerLog;

    /**
     * Logs for the workcleaner service
     */
    private IDeployerLog workCleanerLog;

    /**
     * Default constructor
     * @param bundleContext The context
     */
    public AddonDeployerComponent(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    /**
     * Start the service
     * @throws org.ow2.jonas.service.ServiceException
     */
    @Override
    protected void doStart() throws ServiceException {

        //ceate parent directory of the addons log file, if !exist
        File logFile = new File(AddonUtil.getAddonLogFile(this.serverProperties));
        if (!logFile.getParentFile().exists()) {
            logFile.getParentFile().mkdirs();
        }

        //create the addons log file
        try {
            logFile.createNewFile();
        } catch (IOException e) {
            logger.error("Cannot create the log file for addon deployer");
        }

        //create parent directory of the workcleaner log file if !exist
        File workCleanerLogFile = new File(AddonUtil.getWorkCleanerLogFile(this.serverProperties));
        if (!workCleanerLogFile.exists()) {
            workCleanerLogFile.getParentFile().mkdirs();
            try {
                workCleanerLogFile.createNewFile();
            } catch (IOException e) {
                logger.error("Cannot create the cleaner log file for addon deployer");
            }
        }

        try {
            this.deployerLog = new AddonDeployerLog(logFile);
        } catch (DeployerLogException e) {
            logger.error("Cannot create an addons log");
        }

        try {
            this.workCleanerLog = new DeployerLog(workCleanerLogFile);
        } catch (DeployerLogException e) {
            logger.error("Cannot create a workcleaner log for the addons");
        }

        //Create a new AddonDeployer
        this.addonDeployer = new AddonDeployerImpl(this.serverProperties, this.deployableHelper, this.deployerLog,
                this.deployerManager, this.bundleContext,  this.serviceManager,
                this.workCleanerLog);

        //register the addon deployer
        this.deployerManager.register(this.addonDeployer);
    }

    /**
     * Stop the service
     * @throws org.ow2.jonas.service.ServiceException
     */
    @Override
    protected void doStop() throws ServiceException {
        this.deployerManager.unregister(this.addonDeployer);
    }

    /**
     * @param deployerManager the {@link org.ow2.util.ee.deploy.api.deployer.IDeployerManager} to set
     */
    public void setDeployerManager(final IDeployerManager deployerManager) {
        this.deployerManager = deployerManager;
    }

    /**
     * @param serverProperties ServerProperties to set
     */
    public void setServerProperties(final ServerProperties serverProperties) {
        this.serverProperties = serverProperties;
    }

    /**
     * @param deployableHelper The DeployableHelper which analyze an archive and build the associated Deployable object
     */
    public void setDeployableHelper(final IDeployableHelper deployableHelper) {
        this.deployableHelper = deployableHelper;
    }

    /**
     * @param serviceManager The service manager to registered
     */
    public void registerServiceManager(final ServiceManager serviceManager) {
        this.serviceManager = serviceManager;
    }

    /**
     * Unregistered the service manager
     */
    public void unregisterServiceManager() {
        this.serviceManager = null;
    }

    /**
     * Method called when the workCleanerService is bound to the component.
     * @param workCleanerService the workCleanerService reference
     */
    protected void setWorkCleanerService(final WorkCleanerService workCleanerService) {
        CleanTask cleanTask = new AddonCleanTask(this.addonDeployer, this.workCleanerLog);
        workCleanerService.registerTask(cleanTask);
        workCleanerService.executeTasks();
    }
}
