/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: RegistryInjector.java 19805 2010-05-02 11:27:19Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.component;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.ow2.jonas.camel.service.api.ICamelService;

public class RegistryInjector {

    private static final String REGISTRY_PREFIX = "registry";

    private Map<String, String> replacements = new HashMap<String, String>();

    private File conf;

    private List<String> registryFilesContent;

    public RegistryInjector(final Map<String, String> replacements) throws Exception {
        final String JONAS_BASE = System.getProperty("jonas.base");
        if (JONAS_BASE == null) {
            throw new RuntimeException("JONAS_BASE not defined");
        }
        this.conf = new File(JONAS_BASE, "conf").getAbsoluteFile();

        // Read all registry properties, which will be used for replacing
        // variables (starting with $) in the given XML files
        for (File file : this.conf.listFiles()) {
            final String fileName = file.getName();
            if (fileName.startsWith(RegistryInjector.REGISTRY_PREFIX) && fileName.endsWith(".properties")) {
                InputStream is = new FileInputStream(file);
                try {
                    Properties registryProperties = new Properties();
                    registryProperties.load(is);

                    for (Map.Entry<Object, Object> registryProperty : registryProperties.entrySet()) {
                        String key = (String) registryProperty.getKey();
                        String value = (String) registryProperty.getValue();
                        this.replacements.put(key, value);
                    }
                } finally {
                    is.close();
                    is = null;
                    System.gc();
                }
            }
        }
        if (replacements != null) {
            this.replacements.putAll(replacements);
        }

        // Read all registry XML files and inject properties
        this.registryFilesContent = new ArrayList<String>();
        for (File file : this.conf.listFiles()) {
            final String fileName = file.getName();
            if (fileName.startsWith(RegistryInjector.REGISTRY_PREFIX) && fileName.endsWith(".xml")) {
                String registryString;

                // Read the file
                FileReader fr = new FileReader(file);
                try {
                    BufferedReader br = new BufferedReader(fr);
                    try {
                        StringBuilder sb = new StringBuilder();
                        String line;
                        while ((line = br.readLine()) != null) {
                            sb.append(line);
                            sb.append('\n');
                        }
                        registryString = sb.toString();
                    } finally {
                        br.close();
                        br = null;
                        System.gc();
                    }
                } finally {
                    fr.close();
                    fr = null;
                    System.gc();
                }

                for (Map.Entry<String, String> replacement : this.replacements.entrySet()) {
                    registryString = registryString.replace("$" + replacement.getKey(), replacement.getValue());
                }

                this.registryFilesContent.add(registryString);
            }
        }
    }

    public List<String> getRegistryFilesContent() {
        return this.registryFilesContent;
    }

    public void injectRegistry(final ICamelService camelService, final String camelContextName) throws Exception {
        for (String registryString : this.registryFilesContent) {
            ByteArrayInputStream bais = new ByteArrayInputStream(registryString.getBytes());
            camelService.addRegistry(bais, camelContextName);
        }
    }
}
