/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelServlet.java 19961 2010-06-08 17:00:12Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.component;

import java.io.InputStream;
import java.io.IOException;
import javax.servlet.Servlet;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;

import org.apache.camel.CamelContext;
import org.apache.camel.Exchange;
import org.apache.camel.Producer;

/**
 * Generic CAMEL servlet, that can be for example used for REST endpoints. It
 * will map all request parameters to the Camel exchange's in message headers.
 * Note that all servlet headers use {@link ServletRequest.getParameterMap()},
 * therefore the keys in the parameter map are of type String and the values in
 * the parameter map are of type String array.<br/>
 * <br/>
 * In order to use it:
 * <ul>
 * <li>Decide of a servlet path (for example, <code>/myServlet</code>) and an
 * endpoint name (for example, <code>direct:myServlet</code>).
 * <li>Register the servlet: <code><pre> import org.osgi.service.http.HttpService;
 *
 * ...
 *
 * HttpService httpService = getService(HttpService.class);
 * HttpContext httpContext = httpService.createDefaultHttpContext();
 * Dictionary<String, String> initParams = new Hashtable<String, String>();
 * initParams.put("matchOnUriPrefix", "false");
 * initParams.put("servlet-name", SERVLET_PATH);
 * httpService.registerServlet(SERVLET_PATH,
 *     new CamelServlet(this.getContext(), ENDPOINT_NAME),
 *     initParams, httpContext);</pre></code></li>
 * <li>Handle incoming messages, exactly as you would do with any other CAMEL
 * route:<br/>
 * <code>this.from(ENDPOINT_NAME).process(new MyProcessor());</code></li>
 * </ul>
 */
public class CamelServlet implements Servlet {

    private CamelContext camelContext;

    private String camelDestination;

    private ServletConfig servletConfig;

    /**
     * Creates a new CamelServlet instance.
     *
     * @param camelContext CAMEL context to send messages to.
     * @param camelDestination Name of the CAMEL endpoint on the camelContext
     */
    public CamelServlet(CamelContext camelContext, String camelDestination) {
        this.camelContext = camelContext;
        this.camelDestination = camelDestination;
    }

    public void init(ServletConfig servletConfig) throws ServletException {
        this.servletConfig = servletConfig;
    }

    public ServletConfig getServletConfig() {
        return this.servletConfig;
    }

    public void service(ServletRequest request, ServletResponse rsp) throws ServletException, IOException {
        final ClassLoader oldTCCL = Thread.currentThread().getContextClassLoader();

        try {
            // Change the TCCL in order to call, for example, CXF endpoints
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());

            HttpServletResponse response = (HttpServletResponse) rsp;

            Producer producer = this.camelContext.getEndpoint(camelDestination).createProducer();
            Exchange exchange = producer.createExchange();
            exchange.getIn().setHeaders(request.getParameterMap());
            exchange.getIn().setBody(request.getInputStream());
            producer.process(exchange);

            // Extract file name if exists
            String filename = exchange.getOut().getHeader(Exchange.FILE_NAME, String.class);
            if (filename == null) {
                filename = exchange.getIn().getHeader(Exchange.FILE_NAME, String.class);
            }
            if (filename != null) {
                response.setHeader("filename", filename);
            }

            if (exchange.getOut().getBody() == null) {
                return;
            }

            // Body is not null, is it a String ?
            Object bodyString = exchange.getOut().getBody();
            if (bodyString instanceof String) {
                response.setContentType("text");
                response.getWriter().write((String) bodyString);
                return;
            }

            // Body is not a String, it is binary
            response.setContentType("application/octet-stream");

            // Body is not a String, is it an InputStream ?
            InputStream bodyStream = exchange.getOut().getBody(InputStream.class);
            if (bodyStream != null) {
                try {
                    int read;
                    while ((read = bodyStream.read()) != -1) {
                        response.getOutputStream().write(read);
                    }
                } finally {
                    bodyStream.close();
                    return;
                }
            }

            // Body is not an InputStream, is it a byte array ?
            byte[] bodyByteArray = exchange.getOut().getBody(byte[].class);
            if (bodyByteArray == null) {
                throw new NullPointerException(
                    "Request body is not a String, InputStream or byte array: "
                    + exchange.getOut().getBody().getClass());
            }
            response.getOutputStream().write(bodyByteArray);
        } catch (Exception e) {
            throw new ServletException("Cannot create CAMEL exchange", e);
        } finally {
             Thread.currentThread().setContextClassLoader(oldTCCL);
        }
    }

    public String getServletInfo() {
        return this.toString();
    }

    public void destroy() {
        // Nothing to do
    }

}
