/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: HttpClientProcessor.java 20017 2010-06-23 09:23:45Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.component;

import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Map;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;

/**
 * HTTP client processor, which can replace the CAMEL http processor. We need
 * this processor as <code>camel-http</code> has some weird dependencies, most
 * notably and old version of <code>HTTPClient</code>, which doesn't behave well
 * with OSGi environments.<br/>
 * <br/>
 * It does simple HTTP GET requests and adds all CAMEL headers to the GET
 * request's arguments.<br/>
 * <br/>
 * It is used exactly as any other processor:
 * <ul>
 * <code><pre>
 * // Example: search on Google the file name of the incoming file
 * this.from("file:///camel/test-google-requests")
 * // Google uses the HTTP GET variable q for the search query
 * .setHeader("q", this.header(Exchange.FILE_NAME))
 * .process(new HttpClientProcessor("http://www.google.com/"))
 * // Write response to an HTML file
 * .to("file:///camel/test-google-responses/?fileName=${file:name}.html");
 * </pre></code>
 * </ul>
 */
public class HttpClientProcessor implements Processor {

    private String httpUrl;

    public HttpClientProcessor(final String httpUrl) {
        this.httpUrl = httpUrl;
    }

    public void process(final Exchange exchange) throws Exception {
        StringBuilder urlBuilder = new StringBuilder(this.httpUrl);
        boolean appendQuestionMark = !this.httpUrl.contains("?");
        for (Map.Entry<String, Object> header : exchange.getIn().getHeaders().entrySet()) {
            appendQuestionMark = this.appendKeyValue(appendQuestionMark, urlBuilder, header.getKey(), header.getValue());
        }
        URL httpUrl = new URL(urlBuilder.toString());
        HttpURLConnection connection = (HttpURLConnection) httpUrl.openConnection(Proxy.NO_PROXY);

        exchange.getOut().getHeaders().putAll(exchange.getIn().getHeaders());
        exchange.getOut().setBody(connection.getInputStream());
    }

    private boolean appendKeyValue(boolean appendQuestionMark, final StringBuilder urlBuilder, final String key,
        final Object value) throws Exception {

        final String charset = "UTF-8";

        if (value != null) {
            if (value instanceof String[]) {
                String[] valueStringArray = (String[]) value;
                for (String valueString : valueStringArray) {
                    appendQuestionMark = this.appendKeyValue(appendQuestionMark, urlBuilder, key, valueString);
                }
            } else {
                if (appendQuestionMark) {
                    urlBuilder.append('?');
                    appendQuestionMark = false;
                } else {
                    urlBuilder.append('&');
                }
                urlBuilder.append(URLEncoder.encode(key, charset));
                urlBuilder.append('=');
                urlBuilder.append(URLEncoder.encode(value.toString(), charset));
            }
        }
        return appendQuestionMark;
    }
}
