/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleCXF.java 17629 2009-06-12 14:54:41Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.samples.camel.example.cxf;

import java.io.InputStream;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.cxf.CxfComponent;
import org.apache.cxf.frontend.ClientProxyFactoryBean;
import org.apache.cxf.message.MessageContentsList;
import org.ow2.jonas.samples.camel.example.cxf.webservice.api.ISayHello;
import org.ow2.jonas.samples.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class ExampleCXF {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The camel service. Injected by the container.
     */
    private ICamelService camelService = null;

    /**
     * The camel context name of the example.
     */
    private String camelContextName = null;

    /**
     * Start the example.
     */
    public void start() throws Exception {

        // Start a new context for the application
        this.camelContextName = this.camelService.startNewContext();
        this.logger.info("Context started");

        // Add entries to the registry
        InputStream input = this.getClass().getClassLoader().getResourceAsStream("cxf-registry.xml");
        this.camelService.addRegistry(input, this.camelContextName);

        // Prepare a route to add in the created context
        RouteBuilder builder = new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                this.from("registry:cxfToFile").process(new Processor() {
                    public void process(final Exchange exchange) throws Exception {
                        MessageContentsList msgList = (MessageContentsList) exchange.getIn().getBody();
                        String name = (String) msgList.get(0);
                        ExampleCXF.this.logger.info("Message : {0}", name);

                    }
                }).convertBodyTo(String.class).to("file://" + System.getProperty("java.io.tmpdir") + "/testWS/JOnAS");
            }
        };

        this.camelService.addComponent("cxf", new CxfComponent(), this.camelContextName);

        // Add the route in the camel context.
        this.camelService.addRoutes(builder, this.camelContextName);

        this.test();
    }

    /**
     * Test the route.
     */
    public void test() {

        ClientProxyFactoryBean factory = new ClientProxyFactoryBean();
        factory.setServiceClass(ISayHello.class);
        factory.setAddress("http://localhost:9000/services/SayHello");
        ISayHello client = (ISayHello) factory.create();

        client.hello("guillaume");
    }

    /**
     * Set the service instance. Initialized by iPOJO.
     * 
     * @param service the Camel service
     */
    public void setCamelService(final ICamelService service) {
        this.camelService = service;
    }

    /**
     * Stop the context when the service or when the bundle is removed.
     * 
     * @param service the Camel service
     */
    public void unsetCamelService(final ICamelService service) throws Exception {
        try {
            service.stop(this.camelContextName);
        } finally {
            this.camelContextName = null;
            this.camelService = null;
        }
    }

}
