/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleCXF.java 19556 2010-03-26 13:42:36Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.example.cxf.route;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.Map;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.cxf.CxfComponent;
import org.apache.camel.component.cxf.CxfConstants;
import org.apache.cxf.frontend.ClientProxyFactoryBean;
import org.apache.cxf.message.MessageContentsList;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.http.HttpService;
import org.ow2.jonas.camel.example.cxf.webservice.api.ISayHello;
import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class ExampleCXF {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The camel service. Injected by the container.
     */
    private ICamelService camelService = null;

    /**
     * The camel context name of the example.
     */
    private String camelContextName = null;

    /**
     * The OSGi platform's bundle context.
     */
    private BundleContext bundleContext;

    /**
     * HTTP port number.
     */
    private String httpPort;

    /**
     * Constructor: saves the BundleContext.
     */
    public ExampleCXF(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    /**
     * Start the route.
     */
    public void start() throws Throwable {
        ServiceReference httpService = this.bundleContext.getServiceReference(HttpService.class.getName());
        if (httpService == null) {
            throw new NullPointerException("HttpService is not registered");
        }
        this.httpPort = (String) httpService.getProperty("org.osgi.service.http.port");

        final ClassLoader bundleCL = this.getClass().getClassLoader();
        final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();

        try {
            // First route definition
            RouteBuilder helloService = new RouteBuilder() {
                @Override
                public void configure() throws Exception {
                    this.from("registry:helloService").process(new Processor() {
                        public void process(final Exchange exchange) throws Exception {
                            MessageContentsList msgList = exchange.getIn().getBody(MessageContentsList.class);
                            String name = (String) msgList.get(0);
                            ExampleCXF.this.logger.info("Received CXF message: {0}", name);

                            MessageContentsList response = new MessageContentsList();
                            response.add("hello, " + name);
                            exchange.getOut().setBody(response);
                        }
                    });
                }
            };

            // Second route definition
            RouteBuilder fileToHelloService = new RouteBuilder() {
                Map<String, String> filenames = new HashMap<String, String>();

                @Override
                public void configure() throws Exception {
                    File source = new File("target/fileToHelloService/requests");
                    source.mkdirs();
                    File target = new File("target/fileToHelloService/responses");
                    target.mkdirs();

                    this.from(source.getAbsoluteFile().toURL().toString() + "?delete=true&readLock=changed").process(
                        new Processor() {
                            public void process(final Exchange exchange) throws Exception {
                                filenames.put(exchange.getExchangeId(), exchange.getIn().getHeader(Exchange.FILE_NAME,
                                    String.class));

                                String message = exchange.getIn().getBody(String.class);

                                exchange.getOut().setHeader(CxfConstants.OPERATION_NAME, "hello");

                                MessageContentsList msgList = new MessageContentsList();
                                msgList.add(0, message);
                                exchange.getOut().setBody(msgList);
                            }
                        }).to("registry:helloService").process(new Processor() {
                        public void process(final Exchange exchange) throws Exception {
                            String filename = filenames.remove(exchange.getExchangeId());
                            exchange.getOut().setHeader(Exchange.FILE_NAME, filename);
                            exchange.getOut().setBody(exchange.getIn().getBody());
                        }
                    }).to(target.getAbsoluteFile().toURL().toString());
                }
            };

            // Start a new Camel context for the routes
            this.camelContextName = this.camelService.startNewContext();
            this.logger.info("Camel context {0} started", this.camelContextName);

            // Add the CXF component
            this.camelService.addComponent("cxf", new CxfComponent(), this.camelContextName);

            // Add entries to the registry
            InputStream input = bundleCL.getResourceAsStream("cxf-registry.xml");
            BufferedReader reader = new BufferedReader(new InputStreamReader(input));
            StringBuilder sb = new StringBuilder();
            String line = null;
            while ((line = reader.readLine()) != null) {
                line = line.replace("$http.port", this.httpPort);
                sb.append(line);
                sb.append('\n');
            }
            reader.close();
            input.close();
            input = new ByteArrayInputStream(sb.toString().getBytes());
            this.camelService.addRegistry(input, this.camelContextName);

            try {
                // Change the TCCL since CXF will be loading classes
                // dynamically and cannot use the bundle CL for this.
                //
                // This is needed for both creating the Camel-CXF route
                // and using the ClientProxyFactoryBean-based client.
                Thread.currentThread().setContextClassLoader(bundleCL);

                // Add the routes in the camel context.
                this.camelService.addRoutes(helloService, this.camelContextName);
                this.camelService.addRoutes(fileToHelloService, this.camelContextName);

                this.test();
            } finally {
                Thread.currentThread().setContextClassLoader(oldCL);
            }
        } catch (Throwable t) {
            this.logger.error("Cannot start ExampleCXF", t);
            throw t;
        }
    }

    /**
     * Test the routes.
     */
    public void test() throws Exception {
        ClientProxyFactoryBean factory = new ClientProxyFactoryBean();
        factory.setServiceClass(ISayHello.class);
        factory.setAddress("http://localhost:" + this.httpPort + "/services/SayHello");
        ISayHello client = (ISayHello) factory.create();

        String response = client.hello("guillaume");
        this.logger.info("Got CXF response: {0}", response);

        File selfTestDir = new File("target/fileToHelloService/requests");
        selfTestDir.mkdirs();
        File selfTest = new File(selfTestDir, "selfTest.txt").getAbsoluteFile();
        FileOutputStream fo = new FileOutputStream(selfTest);
        fo.write("ali".getBytes());
        fo.close();
    }

    /**
     * Stop the route.
     */
    public void stop() throws Exception {
        try {
            this.camelService.stop(this.camelContextName);
        } finally {
            this.camelContextName = null;
            this.camelService = null;
        }
    }

}
