/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009-2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleCXF.java 19967 2010-06-09 13:47:02Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.example.cxf.route;

import java.io.File;
import java.io.FileOutputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.component.cxf.CxfConstants;
import org.apache.cxf.frontend.ClientProxyFactoryBean;
import org.apache.cxf.message.MessageContentsList;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.http.HttpService;
import org.ow2.jonas.camel.component.RouteBuilderComponent;
import org.ow2.jonas.camel.example.cxf.webservice.api.ISayHello;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class ExampleCXF extends RouteBuilderComponent {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * HTTP port number.
     */
    private String httpPort;

    /**
     * Name of files exchanged in CXF. Used by the second route.
     */
    private Map<String, String> filenames = new HashMap<String, String>();

    /**
     * Constructor: saves the BundleContext.
     */
    public ExampleCXF(final BundleContext bundleContext) {
        super(bundleContext);
    }

    /**
     * Adds the OSGi HTTP service port and the
     * <code>target/fileToHelloService</code> directory into
     * {@link RouteBuilderComponent#registryReplacements}, calls
     * {@link RouteBuilderComponent#start()} and launches a self-test.
     */
    @Override
    public void start() throws Throwable {
        ServiceReference httpService = this.bundleContext.getServiceReference(HttpService.class.getName());
        if (httpService == null) {
            throw new NullPointerException("HttpService is not registered");
        }
        this.httpPort = (String) httpService.getProperty("org.osgi.service.http.port");
        this.registryReplacements.put("http.port", this.httpPort);

        File fileToHelloService = new File("target/fileToHelloService");
        fileToHelloService.mkdirs();
        String fileToHelloServicePath = fileToHelloService.getAbsolutePath().replace('\\', '/');
        this.registryReplacements.put("fileToHelloService", fileToHelloServicePath);

        super.start();

        // For the internal test, we need to change the class loader
        //
        // Note that in the CAMEL routes, the class loader is always correct so
        // we don't need to worry about this...
        final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            this.test();
        } finally {
            Thread.currentThread().setContextClassLoader(oldCL);
        }
    }

    /**
     * Add routes to the CAMEL context.
     */
    @Override
    public void configure() throws Exception {
        super.configure();

        // First route: logs the incoming message and replies with a "hello"
        this.from("registry:helloService").process(new Processor() {
            public void process(final Exchange exchange) throws Exception {
                MessageContentsList msgList = exchange.getIn().getBody(MessageContentsList.class);
                String name = (String) msgList.get(0);
                ExampleCXF.this.logger.info("Received CXF message: " + name);

                MessageContentsList response = new MessageContentsList();
                response.add("hello, " + name);
                exchange.getOut().setBody(response);
            }
        });

        // Second route: receives files, sends its content via CXF and writes
        // the response

        this.from("registry:fileToHelloService_requests").process(new Processor() {
            public void process(final Exchange exchange) throws Exception {
                ExampleCXF.this.filenames.put(exchange.getExchangeId(), exchange.getIn().getHeader(Exchange.FILE_NAME,
                    String.class));

                String message = exchange.getIn().getBody(String.class);

                exchange.getOut().setHeader(CxfConstants.OPERATION_NAME, "hello");

                MessageContentsList msgList = new MessageContentsList();
                msgList.add(0, message);
                exchange.getOut().setBody(msgList);
            }
        }).to("registry:helloService").process(new Processor() {
            public void process(final Exchange exchange) throws Exception {
                String filename = ExampleCXF.this.filenames.remove(exchange.getExchangeId());
                exchange.getOut().setHeader(Exchange.FILE_NAME, filename);
                exchange.getOut().setBody(exchange.getIn().getBody());
            }
        }).to("registry:fileToHelloService_responses");
    }

    /**
     * Test the routes.
     */
    public void test() throws Exception {
        ClientProxyFactoryBean factory = new ClientProxyFactoryBean();
        factory.setServiceClass(ISayHello.class);
        factory.setAddress("http://localhost:" + this.httpPort + "/services/SayHello");
        ISayHello client = (ISayHello) factory.create();

        String response = client.hello("guillaume");
        this.logger.info("Got CXF response: " + response);

        File selfTestDir = new File("target/fileToHelloService/requests");
        selfTestDir.mkdirs();
        File selfTest = new File(selfTestDir, "selfTest.txt").getAbsoluteFile();
        FileOutputStream fo = new FileOutputStream(selfTest);
        fo.write("ali".getBytes());
        fo.close();
    }

}
