/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleLoader.java 17629 2009-06-12 14:54:41Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.samples.camel.example.jms;

import java.io.InputStream;

import javax.jms.ConnectionFactory;
import javax.naming.InitialContext;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.jms.JmsComponent;
import org.ow2.jonas.samples.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.springframework.jms.support.destination.JndiDestinationResolver;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class ExampleLoader {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The camel service. Injected by the container.
     */
    private ICamelService camelService = null;

    /**
     * The camel context name of the example.
     */
    private String camelContextName = null;

    /**
     * Start the example.
     */
    public void start() throws Exception {
        // Start a new context for the application
        this.camelContextName = this.camelService.startNewContext();
        this.logger.info("Context started");

        // Add the JORAM component
        JmsComponent joram = new JmsComponent();
        ConnectionFactory connectionFactory;

        connectionFactory = (ConnectionFactory) new InitialContext().lookup("CF");

        joram.setConnectionFactory(connectionFactory);
        JndiDestinationResolver jndiDestinationResolver = new JndiDestinationResolver();
        jndiDestinationResolver.setCache(true);

        joram.setDestinationResolver(jndiDestinationResolver);

        this.camelService.addComponent("joram", joram, this.camelContextName);
        this.logger.debug("JORAM component added");

        // Add the registry entries
        ClassLoader cl = this.getClass().getClassLoader();
        InputStream input = cl.getResourceAsStream("jms-registry.xml");
        this.camelService.addRegistry(input, this.camelContextName);

        // Prepare a route to add in the created context
        RouteBuilder builder = new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                this.from("registry:queue").to("file://" + System.getProperty("java.io.tmpdir") + "/test");

                // set up a listener on the file component
                this.from("file://" + System.getProperty("java.io.tmpdir") + "/test").convertBodyTo(String.class).process(
                    new Processor() {
                        public void process(final Exchange e) {
                            ExampleLoader.this.logger.info("From File: " + e.getIn());
                        }
                    });
            }
        };

        // Add the route in the camel context.
        this.camelService.addRoutes(builder, this.camelContextName);

        // Launch kind of a test.
        this.test();
    }

    /**
     * Test the route.
     */
    public void test() {
        // Get a message producer.
        ProducerTemplate template = this.camelService.getProducerTemplate(this.camelContextName);

        // Send five times a message on a given endpoint. The registry
        // component is a default one provided by the camel service.
        final int interationNumber = 5;

        for (int i = 0; i < interationNumber; i++) {
            template.sendBody("registry:queue", "Test Message: " + i);

        }
    }

    /**
     * Set the service instance. Initialized by iPOJO.
     * 
     * @param service the Camel service
     */
    public void setCamelService(final ICamelService service) {
        this.camelService = service;
    }

    /**
     * Stop the context when the service or when the bundle is removed.
     * 
     * @param service the Camel service
     */
    public void unsetCamelService(final ICamelService service) throws Exception {
        try {
            service.stop(this.camelContextName);
        } finally {
            this.camelContextName = null;
            this.camelService = null;
        }
    }

}
