/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelCxfMtomTest.java 18163 2009-08-03 18:07:06Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.test;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.UUID;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.cxf.CxfComponent;
import org.apache.camel.component.cxf.CxfConstants;
import org.apache.camel.component.file.GenericFile;
import org.apache.cxf.Bus;
import org.apache.cxf.message.MessageContentsList;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.ops4j.pax.exam.CoreOptions;
import org.ops4j.pax.exam.Inject;
import org.ops4j.pax.exam.Option;
import org.ops4j.pax.exam.junit.Configuration;
import org.ops4j.pax.exam.junit.JUnit4TestRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.http.HttpService;
import org.ow2.jonas.camel.service.api.ICamelService;

/**
 * Run the Camel-CXF tests on an OSGi gateway.
 * 
 * @author Guillaume Porcher
 */
@RunWith(JUnit4TestRunner.class)
public class CamelCxfMtomTest {

    /**
     * The bundle context.
     */
    @Inject
    private BundleContext bundleContext;

    private static int TIME_OUT = 20;

    private static final int port = 15742;

    private static final String cxfContext = "/services";

    private static final String busName = "cxfBusName";

    private static final File mtomTest = new File("target/mtom-test");

    private static final File input = new File(CamelCxfMtomTest.mtomTest, "input");

    private static final File output = new File(CamelCxfMtomTest.mtomTest, "output");

    private static org.osgi.service.cm.Configuration configuration;

    private static final int FILE_READ_BUFFER_SIZE = 1024;

    private static final String filename = "someFile.txt";

    private static final String message = "Test Message with a random UUID: " + UUID.randomUUID();

    /**
     * Configuration of the OSGi gateway.
     * 
     * @return the configuration
     */
    @Configuration
    public static Option[] configuration() {
        List<Option> options = new ArrayList<Option>();

        // Felix and iPOJO
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.apache.felix.ipojo").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.osgi.compendium").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.apache.felix.configadmin").versionAsInProject()));

        // Core
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("commons-io").artifactId("commons-io")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId(
            "ow2-bundles-externals-commons-logging").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId("ow2-util-i18n")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId("ow2-util-log")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.osgi").artifactId("javaee-api")
            .versionAsInProject()));

        // Camel
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-core")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-spring")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-osgi")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId(
            "ow2-bundles-externals-jaxb2").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId("camel-service")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework.osgi").artifactId(
            "spring-osgi-io").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework.osgi").artifactId(
            "spring-osgi-core").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework").artifactId("spring")
            .versionAsInProject()));

        // CXF
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.ws.commons.schema").artifactId(
            "XmlSchema").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId(
            "camel-jonas5-cxf").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId(
            "camel-jonas5-jaxws").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.osgi").artifactId("neethi")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.osgi").artifactId("woodstox")
            .versionAsInProject()));

        // Camel-CXF
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId(
            "cxf-servlet-deployer").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-cxf")
            .versionAsInProject()));

        // OSGi HTTPService
        options.add(CoreOptions.systemProperty("org.osgi.service.http.port").value(Integer.toString(CamelCxfMtomTest.port)));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.mortbay.jetty").artifactId("jetty")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.mortbay.jetty").artifactId("jetty-util")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ops4j.pax.web").artifactId("pax-web-bundle")
            .versionAsInProject()));

        return options.toArray(new Option[options.size()]);
    }

    @Test
    public void sendFileViaWebService() throws Exception {
        CamelCxfMtomTest.delete(CamelCxfMtomTest.mtomTest);
        CamelCxfMtomTest.output.mkdirs();
        CamelCxfMtomTest.input.mkdirs();

        this.startCxf();
        this.testSendFileViaWebService();
    }

    private void startCxf() throws Exception {
        // check http service is available
        ServiceReference ref3 = this.bundleContext.getServiceReference(HttpService.class.getName());
        Assert.assertNotNull("Assert http service availability", ref3);

        // create CXF servlet deployer instance
        ServiceReference configurationAdminReference = this.bundleContext.getServiceReference(ConfigurationAdmin.class
            .getName());

        Assert.assertNotNull(configurationAdminReference);
        ConfigurationAdmin confAdmin = (ConfigurationAdmin) this.bundleContext.getService(configurationAdminReference);

        CamelCxfMtomTest.configuration = confAdmin.createFactoryConfiguration(
            "org.ow2.jonas.camel.cxf.servlet.OsgiCXFServletDeployer", null);

        Properties properties = new Properties();
        properties.setProperty("servletPath", CamelCxfMtomTest.cxfContext);
        properties.setProperty("busName", CamelCxfMtomTest.busName);
        CamelCxfMtomTest.configuration.update(properties);

        // wait for ipojo instance creation
        final long stopTime = System.currentTimeMillis() + CamelCxfMtomTest.TIME_OUT * 1000;
        while (this.bundleContext.getServiceReference(Bus.class.getName()) == null && System.currentTimeMillis() < stopTime) {
            Thread.sleep(1000);
        }

        // check CXF bus has been exported
        ServiceReference ref2 = this.bundleContext.getServiceReference(Bus.class.getName());
        Assert.assertNotNull("Assert bus availability", ref2);

        // check CamelService
        ServiceReference ref = this.bundleContext.getServiceReference(ICamelService.class.getName());
        Assert.assertNotNull("Assert availability", ref);

        ICamelService service = (ICamelService) this.bundleContext.getService(ref);
        Assert.assertNotNull("Assert instance", service);

        String contextname = service.startNewContext();
        Assert.assertNotNull("Assert that a new camel context is created", contextname);

        service.addComponent("cxf", new CxfComponent(), contextname);

        // Prepare a route to add in the created context
        RouteBuilder receiveRouteBuilder = new RouteBuilder() {
            @Override
            @SuppressWarnings("deprecation")
            public void configure() throws Exception {
                this.from(CamelCxfMtomTest.input.toURL().toString()).process(new Processor() {
                    public void process(final Exchange exchange) throws Exception {
                        GenericFile<?> msgList = (GenericFile<?>) exchange.getIn().getBody();

                        exchange.getOut().setHeader(CxfConstants.OPERATION_NAME, "sendFile");
                        exchange.getOut().setHeader(CxfConstants.OPERATION_NAMESPACE, IMTOMFileSender.NAMESPACE);

                        MessageContentsList msgListOut = new MessageContentsList();
                        File outFile = new File(msgList.getAbsoluteFilePath());
                        msgListOut.set(0, outFile.getName());
                        byte[] outFileData = CamelCxfMtomTest.readFileAsByteArray(outFile);
                        msgListOut.set(1, outFileData);
                        exchange.getOut().setBody(msgListOut);
                    }
                }).to(
                    "cxf://http://localhost:" + CamelCxfMtomTest.port + CamelCxfMtomTest.cxfContext + "/"
                        + IMTOMFileSender.WSNAME + "?serviceClass=" + IMTOMFileSender.class.getName() + "&dataFormat=POJO&"
                        + "bus=#" + CamelCxfMtomTest.busName);
            }
        };

        // Add the route in the camel context.
        service.addRoutes(receiveRouteBuilder, contextname);

        // Prepare a route to add in the created context
        RouteBuilder sendRouteBuilder = new RouteBuilder() {
            @Override
            @SuppressWarnings("deprecation")
            public void configure() throws Exception {
                this.from(
                    "cxf://http://localhost:" + CamelCxfMtomTest.port + CamelCxfMtomTest.cxfContext + "/"
                        + IMTOMFileSender.WSNAME + "?serviceClass=" + IMTOMFileSender.class.getName() + "&dataFormat=POJO&"
                        + "bus=#" + CamelCxfMtomTest.busName).process(new Processor() {
                    public void process(final Exchange exchange) throws Exception {
                        MessageContentsList msgListIn = (MessageContentsList) exchange.getIn().getBody();

                        Assert.assertNotNull("Assert that the message has content", msgListIn);
                        Assert.assertEquals("Assert message content size", 2, msgListIn.size());

                        String inFile = (String) msgListIn.get(0);
                        Assert.assertEquals(CamelCxfMtomTest.filename, inFile);
                        exchange.getOut().setHeader(Exchange.FILE_NAME, inFile);

                        byte[] inFileData = (byte[]) msgListIn.get(1);
                        Assert.assertNotNull("Assert that the message's fileContents is filled", inFileData);
                        exchange.getOut().setBody(inFileData);
                    }
                }).to(CamelCxfMtomTest.output.toURL().toString()).process(new Processor() {
                    public void process(final Exchange exchange) throws Exception {
                        // The Web service has a null response type
                        exchange.getOut().setBody(null);
                    }
                });
            }
        };

        // Add the route in the camel context.
        service.addRoutes(sendRouteBuilder, contextname);
    }

    private void testSendFileViaWebService() throws Exception {
        File input = new File(CamelCxfMtomTest.input, CamelCxfMtomTest.filename);
        input.createNewFile();
        FileWriter writer = new FileWriter(input);
        try {
            writer.write(CamelCxfMtomTest.message);
        } finally {
            writer.close();
            writer = null;
            System.gc();
        }

        File output = new File(CamelCxfMtomTest.output, CamelCxfMtomTest.filename);

        // Wait for the result to get created
        final long stopTime = System.currentTimeMillis() + CamelCxfMtomTest.TIME_OUT * 1000;
        while (!output.exists() && System.currentTimeMillis() < stopTime) {
            Thread.sleep(1000);
        }
        Assert.assertTrue("Assert that the file is correctly created", output.exists());

        String outputFile = CamelCxfMtomTest.readFileAsString(output);
        Assert.assertEquals(CamelCxfMtomTest.message, outputFile);
    }

    private static byte[] readFileAsByteArray(final File file) throws IOException {
        InputStream inputStream = new FileInputStream(file);
        try {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            try {
                int readBytes;
                byte[] bytes = new byte[CamelCxfMtomTest.FILE_READ_BUFFER_SIZE];

                while ((readBytes = inputStream.read(bytes)) > 0) {
                    outputStream.write(bytes, 0, readBytes);
                }

                return outputStream.toByteArray();
            } finally {
                outputStream.close();
            }
        } finally {
            inputStream.close();
        }
    }

    private static String readFileAsString(final File file) throws IOException {
        StringBuffer fileData = new StringBuffer();
        FileReader fileReader = new FileReader(file);
        try {
            BufferedReader bufferedReader = new BufferedReader(fileReader);
            try {
                int readChars;
                char[] chars = new char[CamelCxfMtomTest.FILE_READ_BUFFER_SIZE];

                while ((readChars = bufferedReader.read(chars)) > 0) {
                    String readData = String.valueOf(chars, 0, readChars);
                    fileData.append(readData);
                }

                return fileData.toString();
            } finally {
                bufferedReader.close();
            }
        } finally {
            fileReader.close();
        }
    }

    @AfterClass
    public static void destroy() throws Exception {
        if (CamelCxfMtomTest.configuration != null) {
            CamelCxfMtomTest.configuration.delete();
        }
    }

    private static void delete(final File dir) {
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    CamelCxfMtomTest.delete(file);
                } else {
                    file.delete();
                }
            }
        }
    }
}
