/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelCxfTest.java 18163 2009-08-03 18:07:06Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.test;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.cxf.CxfComponent;
import org.apache.cxf.Bus;
import org.apache.cxf.frontend.ClientProxyFactoryBean;
import org.apache.cxf.message.MessageContentsList;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.ops4j.pax.exam.CoreOptions;
import org.ops4j.pax.exam.Inject;
import org.ops4j.pax.exam.Option;
import org.ops4j.pax.exam.junit.Configuration;
import org.ops4j.pax.exam.junit.JUnit4TestRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.http.HttpService;
import org.ow2.jonas.camel.service.api.ICamelService;

/**
 * Run the Camel-CXF tests on an OSGi gateway.
 * 
 * @author Guillaume Porcher
 */
@RunWith(JUnit4TestRunner.class)
public class CamelCxfTest {

    /**
     * The bundle context.
     */
    @Inject
    private BundleContext bundleContext;

    private static int TIME_OUT = 20;

    private static final int port = 15742;

    private static final String cxfContext = "/services";

    private static final String busName = "cxfBusName";

    /**
     * Configuration of the OSGi gateway.
     * 
     * @return the configuration
     */
    @Configuration
    public static Option[] configuration() {
        List<Option> options = new ArrayList<Option>();

        // Felix and iPOJO
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.apache.felix.ipojo").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.osgi.compendium").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.apache.felix.configadmin").versionAsInProject()));

        // Core
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("commons-io").artifactId("commons-io")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId(
            "ow2-bundles-externals-commons-logging").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId("ow2-util-i18n")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId("ow2-util-log")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.osgi").artifactId("javaee-api")
            .versionAsInProject()));

        // Camel
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-core")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-spring")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-osgi")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId(
            "ow2-bundles-externals-jaxb2").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId("camel-service")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework.osgi").artifactId(
            "spring-osgi-io").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework.osgi").artifactId(
            "spring-osgi-core").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework").artifactId("spring")
            .versionAsInProject()));

        // CXF
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.ws.commons.schema").artifactId(
            "XmlSchema").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId(
            "camel-jonas5-cxf").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId(
            "camel-jonas5-jaxws").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.osgi").artifactId("neethi")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.osgi").artifactId("woodstox")
            .versionAsInProject()));

        // Camel-CXF
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId(
            "cxf-servlet-deployer").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-cxf")
            .versionAsInProject()));

        // OSGi HTTPService
        options.add(CoreOptions.systemProperty("org.osgi.service.http.port").value(Integer.toString(CamelCxfTest.port)));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.mortbay.jetty").artifactId("jetty")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.mortbay.jetty").artifactId("jetty-util")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ops4j.pax.web").artifactId("pax-web-bundle")
            .versionAsInProject()));

        return options.toArray(new Option[options.size()]);
    }

    /**
     * Gets the service, starts a new camel context, adds a route on this
     * context, tests the route, stops the context.
     */
    @Test
    public void testCxf() throws Exception {
        // check http service is available
        ServiceReference ref3 = this.bundleContext.getServiceReference(HttpService.class.getName());
        Assert.assertNotNull("Assert http service availability", ref3);

        // create CXF servlet deployer instance
        ServiceReference configurationAdminReference = this.bundleContext.getServiceReference(ConfigurationAdmin.class
            .getName());

        Assert.assertNotNull(configurationAdminReference);
        ConfigurationAdmin confAdmin = (ConfigurationAdmin) this.bundleContext.getService(configurationAdminReference);

        org.osgi.service.cm.Configuration configuration = confAdmin.createFactoryConfiguration(
            "org.ow2.jonas.camel.cxf.servlet.OsgiCXFServletDeployer", null);
        try {
            Properties properties = new Properties();
            properties.setProperty("servletPath", CamelCxfTest.cxfContext);
            properties.setProperty("busName", CamelCxfTest.busName);
            configuration.update(properties);

            // wait for ipojo instance creation
            final long stopTime = System.currentTimeMillis() + CamelCxfTest.TIME_OUT * 1000;
            while (this.bundleContext.getServiceReference(Bus.class.getName()) == null && System.currentTimeMillis() < stopTime) {
                Thread.sleep(1000);
            }

            // check CXF bus has been exported
            ServiceReference ref2 = this.bundleContext.getServiceReference(Bus.class.getName());
            Assert.assertNotNull("Assert bus availability", ref2);

            // check CamelService
            ServiceReference ref = this.bundleContext.getServiceReference(ICamelService.class.getName());
            Assert.assertNotNull("Assert availability", ref);

            ICamelService service = (ICamelService) this.bundleContext.getService(ref);
            Assert.assertNotNull("Assert instance", service);

            String contextname = service.startNewContext();
            Assert.assertNotNull("Assert that a new camel context is created", contextname);

            // Prepare a route to add in the created context
            RouteBuilder builder = new RouteBuilder() {
                @Override
                public void configure() throws Exception {
                    this.from(
                        "cxf://http://localhost:" + CamelCxfTest.port + CamelCxfTest.cxfContext + "/SayHello?"
                            + "serviceClass=org.ow2.jonas.camel.test.ISayHello&dataFormat=POJO&" + "bus=#"
                            + CamelCxfTest.busName).process(new Processor() {
                        public void process(final Exchange exchange) throws Exception {
                            MessageContentsList msgList = (MessageContentsList) exchange.getIn().getBody();
                            String name = (String) msgList.get(0);
                            System.out.println("Message : " + name);
                            msgList.clear();
                            msgList.add("coucou " + name);
                            exchange.setOut(exchange.getIn().copy());
                        }
                    });
                }
            };

            service.addComponent("cxf", new CxfComponent(), contextname);

            // Add the route in the camel context.
            service.addRoutes(builder, contextname);

            ClientProxyFactoryBean factory = new ClientProxyFactoryBean();
            factory.setServiceClass(ISayHello.class);
            factory.setAddress("http://localhost:" + CamelCxfTest.port + CamelCxfTest.cxfContext + "/SayHello");
            ISayHello client = (ISayHello) factory.create();

            String res = client.hello("guillaume");
            Assert.assertNotNull(res);
            Assert.assertEquals("coucou guillaume", res);
            service.stop(contextname);
        } finally {
            configuration.delete();
        }
    }
}
