/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelJmsTest.java 18163 2009-08-03 18:07:06Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.test;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.jms.ConnectionFactory;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.camel.Exchange;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.jms.JmsComponent;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.ops4j.pax.exam.CoreOptions;
import org.ops4j.pax.exam.Inject;
import org.ops4j.pax.exam.Option;
import org.ops4j.pax.exam.junit.Configuration;
import org.ops4j.pax.exam.junit.JUnit4TestRunner;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.ow2.jonas.camel.service.api.ICamelService;

/**
 * Run the tests on an OSGi gateway.
 * 
 * @author Guillaume Renault
 */
@RunWith(JUnit4TestRunner.class)
public class CamelJmsTest {

    /**
     * The bundle context.
     */
    @Inject
    private BundleContext bundleContext;

    /**
     * Java tmp dir.
     */
    private static final String tmpDir = System.getProperty("java.io.tmpdir");

    private static int TIME_OUT = 20;

    /**
     * Configuration of the OSGi gateway.
     * 
     * @return the configuration
     */
    @Configuration
    public static Option[] configuration() {
        List<Option> options = new ArrayList<Option>();

        // Felix and iPOJO
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.apache.felix.ipojo").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.osgi.compendium").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId(
            "org.apache.felix.configadmin").versionAsInProject()));

        // Core
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("commons-io").artifactId("commons-io")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId(
            "ow2-bundles-externals-commons-logging").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId("ow2-util-i18n")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId("ow2-util-log")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.osgi").artifactId("javaee-api")
            .versionAsInProject()));

        // Camel
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.jonas.camel").artifactId("camel-service")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-core")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-spring")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-osgi")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework.osgi").artifactId(
            "spring-osgi-io").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework.osgi").artifactId(
            "spring-osgi-core").versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.springframework").artifactId("spring")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.ow2.bundles").artifactId(
            "ow2-bundles-externals-jaxb2").versionAsInProject()));

        // Camel-JMS
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.camel").artifactId("camel-jms")
            .versionAsInProject()));
        options.add(CoreOptions.provision(CoreOptions.mavenBundle().groupId("org.apache.activemq").artifactId("activemq-core")
            .versionAsInProject()));

        return options.toArray(new Option[options.size()]);
    }

    /**
     * Gets the service, starts a new camel context, adds a route on this
     * context, tests the route, stops the context.
     */
    @Test
    public void isServiceStarted() throws Exception {
        ServiceReference ref = this.bundleContext.getServiceReference(ICamelService.class.getName());
        Assert.assertNotNull("Assert Availability", ref);

        ICamelService service = (ICamelService) this.bundleContext.getService(ref);
        Assert.assertNotNull("Assert instance", service);

        String contextname = service.startNewContext();
        Assert.assertNotNull("Assert that a new camel context is created", contextname);

        ConnectionFactory connectionFactory = new ActiveMQConnectionFactory("vm://localhost?broker.persistent=false");

        service.addComponent("test-jms", JmsComponent.jmsComponentAutoAcknowledge(connectionFactory), contextname);

        RouteBuilder builder = new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                this.from("test-jms:queue:Queue").setHeader(Exchange.FILE_NAME, this.constant("report.txt")).to(
                    "file://" + CamelJmsTest.tmpDir + "/test");

            }
        };

        // Add the route in the camel context.
        service.addRoutes(builder, contextname);

        // Get a message producer.
        ProducerTemplate template = service.getProducerTemplate(contextname);

        // Send five times a message on a given endpoint. The registry component
        // is a default one provided by the camel service.

        String message = "Test Message";

        template.sendBody("test-jms:queue:Queue", message);

        File result = new File(CamelJmsTest.tmpDir + "/test/report.txt");

        // active wait
        final long stopTime = System.currentTimeMillis() + CamelJmsTest.TIME_OUT * 1000;
        while (!result.exists() && System.currentTimeMillis() < stopTime) {
            Thread.sleep(1000);
        }

        Assert.assertTrue("Assert that the file is correctly created", result.exists());

        result.delete();

        service.stop(contextname);
    }
}
