/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelRouteBase.java 18096 2009-07-29 13:12:22Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.route.base;

import org.apache.camel.builder.RouteBuilder;
import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Abstract class : the iPOJO should be implement this class to use Camel
 * Service and add Route
 * 
 * @author chahinem
 */
public abstract class CamelRouteBase {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The camel service. Injected by the container.
     */
    private ICamelService camelService = null;

    /**
     * The camel context name of the example.
     */
    private String camelContextName = null;

    /**
     * Routes to build.
     */
    private RouteBuilder routeBuilder;

    public CamelRouteBase(final RouteBuilder routeBuilder) {
        this.routeBuilder = routeBuilder;
    }

    /**
     * Start the route bundle.
     */
    public final void start() throws Exception {
        // Start a new context for the application
        this.camelContextName = this.camelService.startNewContext();
        this.logger.info("Context started");

        // If necessary, configure the camel context.
        this.configureCamel(this.camelService, this.camelContextName);

        // Add the route in the camel context.
        this.camelService.addRoutes(this.routeBuilder, this.camelContextName);
    }

    /**
     * Adds extra configuration to the camel instance, for example a JNDI
     * registry or some registry entries.
     * 
     * @param camelService camel service.
     * @param camelContextName name of the current camel context in the camel
     *        service.
     */
    protected abstract void configureCamel(ICamelService camelService, String camelContextName);

    /**
     * Set the service instance. Initialized by iPOJO.
     * 
     * @param service the Camel service
     */
    @SuppressWarnings("unused")
    private void setCamelService(final ICamelService service) {
        this.camelService = service;
    }

    /**
     * Stop the context when the service or when the bundle is removed.
     * 
     * @param service the Camel service
     */
    @SuppressWarnings("unused")
    private void unsetCamelService(final ICamelService service) throws Exception {
        try {
            service.stop(this.camelContextName);
        } finally {
            this.camelContextName = null;
            this.camelService = null;
        }
    }
}
