/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: OSGIRegistry.java 17719 2009-06-18 14:15:10Z porcherg $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.samples.camel.registry;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.camel.spi.Registry;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;

/**
 * Implementation of {@link Registry} for looking OSGI services.
 * 
 * @author Guillaume Porcher
 */
public class OSGIRegistry implements Registry {

    /**
     * the bundle context.
     */
    private final BundleContext bundleContext;

    /**
     * Default constructor.
     * 
     * @param bundleContext the current bundle context
     */
    public OSGIRegistry(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    /**
     * Looks for an OSGI service exporting a name property equals to "name".
     * 
     * @see org.apache.camel.spi.Registry#lookup(String)
     */
    public Object lookup(final String name) {
        return this.lookup(name, null);
    }

    /**
     * Looks for an OSGI service exporting a name property equals to "name" and
     * with class clazz.
     * 
     * @see org.apache.camel.spi.Registry#lookup(java.lang.String,
     *      java.lang.Class)
     */
    public <T> T lookup(final String name, final Class<T> clazz) {
        return this.lookupByType(clazz).get(name);
    }

    /**
     * Looks for an OSGI service with class clazz.
     * 
     * @see org.apache.camel.spi.Registry#lookupByType(java.lang.Class)
     */
    @SuppressWarnings("unchecked")
    public <T> Map<String, T> lookupByType(final Class<T> clazz) {
        try {
            ServiceReference[] refs = this.bundleContext.getServiceReferences(clazz == null ? null : clazz.getName(), null);
            if (refs != null && refs.length >= 0) {
                Map<String, T> result = new HashMap<String, T>();
                for (ServiceReference ref : refs) {
                    result.put((String) ref.getProperty("name"), (T) this.bundleContext.getService(ref));
                }
                return result;
            }
        } catch (InvalidSyntaxException e) {
        }
        return Collections.EMPTY_MAP;
    }

}
