/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ICamelService.java 17427 2009-05-20 12:12:28Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.samples.camel.service.api;

import java.io.InputStream;
import java.util.List;

import org.apache.camel.Component;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.model.RouteDefinition;

/**
 * Interface of the Camel Service.
 * 
 * @author Guillaume Renault
 */
public interface ICamelService {

    /**
     * Start a new camel context.
     * 
     * @return the camel context name.
     * @throws Exception If starting the context fails.
     */
    String startNewContext() throws Exception;

    /**
     * Stop an instance of Camel.
     * 
     * @param name the camel context name to stop.
     * @throws Exception If stopping any CAMEL console fails.
     */
    void stop(final String name) throws Exception;

    /**
     * @param builder the route to add.
     * @param camelContextName the camel context name on which the route has to
     *        be set.
     * @throws Exception If adding route fails.
     */
    void addRoutes(final RouteBuilder builder, final String camelContextName) throws Exception;

    /**
     * Get the list of existing routes.
     * 
     * @param camelContextName the identifier of the camel context to check.
     * @return the list of existing routes for the given camel context..
     */
    List<RouteDefinition> getRoutes(final String camelContextName);

    /**
     * Get a producer template for the given camel context.
     * 
     * @param camelContextName the camel context name on which the producer has
     *        to be taken.
     * @return the producer template.
     */
    ProducerTemplate getProducerTemplate(final String camelContextName);

    /**
     * Add values to the registry component.
     * 
     * @param input the stream that contains the entries to add.
     * @param camelContextName the camel context name that contains the registry
     *        to update.
     */
    void addRegistry(final InputStream input, final String camelContextName) throws Exception;

    /**
     * Remove entries from the registry component.
     * 
     * @param input the input stream that contains all the entries to remove
     * @param camelContextName the camel context name to remove the entries on.
     */
    void removeRegistry(final InputStream input, final String camelContextName);

    /**
     * Add a component on the given camel context.
     * 
     * @param componentName the component name.
     * @param component the component.
     * @param camelContextName the camel name context.
     */
    void addComponent(final String componentName, final Component component, final String camelContextName);

    // /**
    // * Add an endpoint on specific Camel context.
    // *
    // * @param uri
    // * the URI to create
    // * @param endpoint
    // * the endpoint to add.
    // * @param camelContextName
    // * the Camel context on wich the Endpoint will be set.
    // */
    // void addEndpoint(final String uri, final Endpoint<? extends Exchange>
    // endpoint, final String camelContextName);
}
