/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelService.java 17427 2009-05-20 12:12:28Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.samples.camel.service.impl;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.camel.Component;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.model.RouteDefinition;
import org.apache.camel.osgi.CamelContextFactory;
import org.osgi.framework.BundleContext;
import org.ow2.jonas.samples.camel.service.api.ICamelService;
import org.ow2.jonas.samples.camel.wrapper.CamelWrapper;
import org.ow2.jonas.samples.camel.wrapper.ICamelWrapper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Camel Wrapper. Main class of the Resource Adapter
 * 
 * @author Guillaume Renault
 */
public class CamelService implements ICamelService {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(CamelService.class);

    /**
     * Camel context factory.
     */
    private CamelContextFactory camelFactory = null;

    /**
     * Map of all of the camel contexts.
     */
    private Map<String, ICamelWrapper> camelContexts = null;

    /**
     * Constructor, needed to get the bundle context.
     * 
     * @param context the bundle context.
     */
    public CamelService(final BundleContext context) {
        this.camelFactory = new CamelContextFactory();
        this.camelFactory.setBundleContext(context);
    }

    /**
     * Initialize the bundle. Called by iPOJO when the bundle is added in the
     * OSGi environment.
     */
    public void initialize() {
        // set the thread context class loader.
        Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());

        this.camelContexts = new HashMap<String, ICamelWrapper>();
        this.logger.info("Camel service started");
    }

    /**
     * Stop the bundle. Called by iPOJO when the bundle is removed from the OSGi
     * environment.
     * 
     * @throws Exception If stopping any CAMEL console fails.
     */
    public void stopAll() throws Exception {
        Map<String, ICamelWrapper> camelContexts = new HashMap<String, ICamelWrapper>(this.camelContexts);
        this.camelContexts = new HashMap<String, ICamelWrapper>();
        for (Map.Entry<String, ICamelWrapper> entry : camelContexts.entrySet()) {
            entry.getValue().stopContext();
            this.logger.debug("The camel context {0} has been stopped.", entry.getKey());
        }
        this.logger.info("Camel service stopped");
    }

    /**
     * Start a new camel context.
     * 
     * @return the name of the camel context.
     */
    public String startNewContext() throws Exception {
        this.logger.info("Starting a new camel context");

        CamelWrapper camelThread = new CamelWrapper(this.camelFactory);

        // set the camel thread class loader to be able to access
        // configurations files
        // camelThread.setContextClassLoader(this.getClass().getClassLoader());

        // put the Thread in the thread list.
        this.camelContexts.put(camelThread.getCamelContextName(), camelThread);

        // set the thread name
        camelThread.setName(camelThread.getCamelContextName());

        // start the context.
        camelThread.run();

        this.logger.debug("A new Camel context has been created with the name {0}", camelThread.getCamelContextName());

        return camelThread.getCamelContextName();

    }

    /**
     * Stop an instance of Camel.
     * 
     * @param name the camel context name to stop.
     * @throws Exception If stopping any CAMEL console fails.
     */
    public void stop(final String name) throws Exception {
        if (this.camelContexts.containsKey(name)) {
            this.camelContexts.get(name).stopContext();
            this.camelContexts.remove(name);

            this.logger.debug("The camel context {0} has been stopped.", name);
        }
    }

    /**
     * @param builder the route to add.
     * @param camelContextName the camel context name on which the route has to
     *        be set.
     * @throws Exception If adding route fails.
     */
    public void addRoutes(final RouteBuilder builder, final String camelContextName) throws Exception {
        this.camelContexts.get(camelContextName).addRoutes(builder);
        this.logger.debug("A route has been added on the following Camel context : {0}", camelContextName);
    }

    /**
     * Get the list of existing routes.
     * 
     * @param camelContextName the identifier of the camel context to check.
     * @return the list of existing routes for the given camel context..
     */
    public List<RouteDefinition> getRoutes(final String camelContextName) {
        return this.camelContexts.get(camelContextName).getRoutes();
    }

    /**
     * Get a producer template for the given camel context.
     * 
     * @param camelContextName the camel context name on which the producer has
     *        to be taken.
     * @return the producer template.
     */
    public ProducerTemplate getProducerTemplate(final String camelContextName) {
        this.logger.debug("Create a new producer template on the following Camel context : {0}", camelContextName);
        return this.camelContexts.get(camelContextName).getProducerTemplate();
    }

    /**
     * Add values to the registry component.
     * 
     * @param input the stream that contains the entries to add.
     * @param camelContextName the camel context name that contains the registry
     *        to update.
     */
    public void addRegistry(final InputStream input, final String camelContextName) throws Exception {
        this.logger.debug("Add entries to the Registry component of the following Camel context : {0}", camelContextName);
        this.camelContexts.get(camelContextName).addToTheRegistry(input);
    }

    /**
     * Remove entries from the registry component.
     * 
     * @param input the input stream that contains all the entries to remove
     * @param camelContextName the camel context name to remove the entries on.
     */
    public void removeRegistry(final InputStream input, final String camelContextName) {
        this.camelContexts.get(camelContextName).removeFromTheRegistry(input);
        this.logger
            .debug("Some entries were removed from the registry of the following camel context : {0}.", camelContextName);
    }

    /**
     * Add a component on the given camel context.
     * 
     * @param componentName the component name.
     * @param component the component.
     * @param camelContextName the camel name context.
     */
    public void addComponent(final String componentName, final Component component, final String camelContextName) {
        // register the component on the camel context.
        this.camelContexts.get(camelContextName).addComponent(componentName, component);

        this.logger.debug("Component {0} added on the Camel context {1}", componentName, camelContextName);

    }

}
