/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelWrapper.java 18099 2009-07-29 13:50:32Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.wrapper;

import java.io.InputStream;
import java.util.List;
import java.util.Set;

import javax.management.AttributeNotFoundException;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;

import org.apache.camel.CamelContext;
import org.apache.camel.Component;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.impl.DefaultCamelContext;
import org.apache.camel.model.RouteDefinition;
import org.apache.camel.osgi.CamelContextFactory;
import org.ow2.jonas.camel.registry.impl.file.FileRegistryComponent;
import org.ow2.jonas.camel.registry.osgi.OSGiRegistry;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Camel wrapper. A camel context is wrapped here.
 * 
 * @author Guillaume Renault
 */
public class CamelWrapper implements ICamelWrapper {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * Main camel instance.
     */
    private CamelContext camel = null;

    /**
     * The file registry component.
     */
    private FileRegistryComponent registry = null;

    /**
     * Name of the camel context.
     */
    private String name;

    /**
     * Constructor. Create a new Camel Context. The bundle context has to be
     * set.
     * 
     * @param bundleContext the OSGi bundle context.
     */
    public CamelWrapper(final CamelContextFactory factory) {
        DefaultCamelContext ctx = factory.createContext();
        ctx.setRegistry(new OSGiRegistry(factory.getBundleContext()));
        this.camel = ctx;
    }

    /**
     * Gets the camel context name.
     * 
     * @return the camel context name.
     */
    public String getCamelContextName() {
        return this.camel.getName();
    }

    /**
     * Main method.
     */
    public void run() throws Exception {
        while (true) {
            try {
                MBeanServer mBeanServer = null;
                ObjectName j2eeServer = null;
                for (Object o : MBeanServerFactory.findMBeanServer(null)) {
                    MBeanServer currentServer = (MBeanServer) o;
                    Set<?> j2eeServerNames = currentServer.queryNames(new ObjectName("*:j2eeType=J2EEServer,*"), null);
                    if (j2eeServerNames.size() > 0) {
                        j2eeServer = (ObjectName) j2eeServerNames.iterator().next();
                        mBeanServer = currentServer;
                        break;
                    }
                }

                if (mBeanServer != null) {
                    if (mBeanServer.getAttribute(j2eeServer, "state").toString().contains("running")) {
                        break;
                    }
                } else {
                    break;
                }
            } catch (AttributeNotFoundException ignored) {
                // Non-Java-EE compliant server (might be JOnAS 4)
                this.logger.debug("Ignore exception {0}, as we are not in a java-EE compliant environment", ignored);
                // We may be on a simple OSGi gateway, so no need to wait
                // anymore.
                break;
            }

            // Server not started, wait a bit and try again
            Thread.sleep(1000);
        }

        this.camel.start();
    }

    /**
     * Stop the camel context.
     */
    public void stopContext() throws Exception {
        this.camel.stop();
    }

    /**
     * Add a route.
     * 
     * @param builder the route to add.
     */
    public void addRoutes(final RouteBuilder builder) throws Exception {
        this.logger.debug("Adding a route to the context");
        this.camel.addRoutes(builder);
    }

    /**
     * @return the list of deployed routes on the camel context.
     */
    public List<RouteDefinition> getRoutes() {
        return this.camel.getRouteDefinitions();
    }

    /**
     * Add values to the registy.
     * 
     * @param inputStream the input stream containing the entire xml file that
     *        defines the entries to add.
     */
    public void addToTheRegistry(final InputStream inputStream) throws Exception {
        if (this.registry == null) {
            this.registry = new FileRegistryComponent();
            this.addComponent("registry", this.registry);
            this.logger.debug("Registry component added");
        }
        this.registry.addValues(inputStream);
    }

    /**
     * Remove values from the registy.
     * 
     * @param inputStream the input stream containing the entire xml file that
     *        defines the entries to remove.
     */
    public void removeFromTheRegistry(final InputStream inputStream) {
        if (this.registry == null) {
            throw new IllegalStateException(
                "The registry has not been initiated with value yet. Removing values is therefore forbidden.");
        } else {
            this.registry.removeValues(inputStream);
        }
    }

    /**
     * Get a producer template for the camel context.
     * 
     * @return a producer for the camel context.
     */
    public ProducerTemplate getProducerTemplate() {
        return this.camel.createProducerTemplate();
    }

    /**
     * Add a component on the given camel context.
     * 
     * @param componentName the component name.
     * @param component the component.
     */
    public void addComponent(final String componentName, final Component component) {
        component.setCamelContext(this.camel);
        this.camel.addComponent(componentName, component);
    }

    /**
     * Get the name of the camel context.
     * 
     * @return the name of the camel context.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Set the name of the camel context.
     * 
     * @param name the name of the camel context.
     */
    public void setName(final String name) {
        this.name = name;
    }

}
