/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008-2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelWrapper.java 19913 2010-06-02 16:00:27Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.wrapper;

import java.io.InputStream;
import java.util.List;
import java.util.Set;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;

import org.apache.camel.CamelContext;
import org.apache.camel.Component;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.ref.RefComponent;
import org.apache.camel.impl.DefaultCamelContext;
import org.apache.camel.model.RouteDefinition;
import org.apache.camel.osgi.CamelContextFactory;
import org.ow2.jonas.camel.registry.osgi.OSGiRegistry;
import org.ow2.jonas.camel.registry.xml.RegistryWrapper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Camel wrapper. A camel context is wrapped here.
 * 
 * @author Guillaume Renault
 */
public class CamelWrapper implements ICamelWrapper {

    /**
     * Logger.
     */
    private final Log logger = LogFactory.getLog(this.getClass());

    /**
     * Main camel instance.
     */
    private CamelContext camel = null;

    /**
     * Name of the camel context.
     */
    private String name;

    /**
     * The camel registry wrapper.
     */
    private RegistryWrapper registryWrapper;

    /**
     * Constructor, creates a new Camel context.
     * 
     * @param factory Camel context factory to use.
     */
    public CamelWrapper(final CamelContextFactory factory) {
        final DefaultCamelContext ctx = factory.createContext();
        OSGiRegistry osgiRegistry = new OSGiRegistry(factory.getBundleContext());
        ctx.setRegistry(osgiRegistry);
        // for compatibility with previous registry only, routes beginning
        // with registry: should be associate with Camel's RefComponent
        ctx.addComponent("registry", new RefComponent());
        // create the wrapper
        this.registryWrapper = new RegistryWrapper();
        // Create the registry and assign the context to it.
        this.registryWrapper.createRegistryAndAssignToContext(ctx);
        this.camel = ctx;
    }

    /**
     * Gets the camel context name.
     * 
     * @return the camel context name.
     */
    public String getCamelContextName() {
        return this.camel.getName();
    }

    /**
     * Main method.
     */
    public void run() throws Exception {
        while (true) {
            MBeanServer mBeanServer = null;
            ObjectName j2eeServer = null;
            for (final Object o : MBeanServerFactory.findMBeanServer(null)) {
                final MBeanServer currentServer = (MBeanServer) o;
                final Set<?> j2eeServerNames = currentServer.queryNames(new ObjectName("*:j2eeType=J2EEServer,*"), null);
                if (j2eeServerNames.size() > 0) {
                    j2eeServer = (ObjectName) j2eeServerNames.iterator().next();
                    mBeanServer = currentServer;
                    break;
                }
            }

            if (mBeanServer == null) {
                this.logger.info("CamelWrapper is running outside a J2EEServer");
                break;
            }

            final String state = mBeanServer.getAttribute(j2eeServer, "state").toString();
            if (state.contains("running")) {
                // Server is running :)
                break;
            } else if (!state.contains("starting")) {
                throw new Exception("Server state is " + state + ", canceling context creation");
            }

            // Server not started, wait a bit and try again
            this.logger.debug("The J2EEServer hosting CamelWrapper is still starting, waiting...");
            Thread.sleep(100);
        }

        this.camel.start();
    }

    /**
     * Stop the camel context.
     */
    public void stopContext() throws Exception {
        this.camel.stop();
    }

    /**
     * Add a route.
     * 
     * @param builder the route to add.
     */
    public void addRoutes(final RouteBuilder builder) throws Exception {
        this.logger.debug("Adding a route to the context");
        this.camel.addRoutes(builder);
    }

    /**
     * @return the list of deployed routes on the camel context.
     */
    public List<RouteDefinition> getRouteDefinitions() {
        return this.camel.getRouteDefinitions();
    }

    /**
     * Get a producer template for the camel context.
     * 
     * @return a producer for the camel context.
     */
    public ProducerTemplate getProducerTemplate() {
        return this.camel.createProducerTemplate();
    }

    /**
     * Add a component on the given camel context.
     * 
     * @param componentName the component name.
     * @param component the component.
     */
    public void addComponent(final String componentName, final Component component) {
        component.setCamelContext(this.camel);
        this.camel.addComponent(componentName, component);
    }

    /**
     * Get the name of the camel context.
     * 
     * @return the name of the camel context.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Set the name of the camel context.
     * 
     * @param name the name of the camel context.
     */
    public void setName(final String name) {
        this.name = name;
    }

    /**
     * @return the wrapped camel instance.
     */
    public CamelContext getCamelContext() {
        return this.camel;
    }

    public void addToTheRegistry(final InputStream inputStream) throws Exception {
        this.registryWrapper.addToTheRegistry(inputStream);
    }

    public void removeFromTheRegistry(final InputStream inputStream) {
        this.registryWrapper.removeFromTheRegistry(inputStream);
    }

    public RegistryWrapper getRegistryWrapper() {
        return this.registryWrapper;
    }

}
