/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.file.remote;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.camel.Exchange;
import org.apache.camel.InvalidPayloadException;
import org.apache.camel.component.file.GenericFile;
import org.apache.camel.component.file.GenericFileEndpoint;
import org.apache.camel.component.file.GenericFileExist;
import org.apache.camel.component.file.GenericFileOperationFailedException;
import org.apache.camel.component.file.remote.RemoteFileConfiguration;
import org.apache.camel.component.file.remote.RemoteFileEndpoint;
import org.apache.camel.component.file.remote.RemoteFileOperations;
import org.apache.camel.util.FileUtil;
import org.apache.camel.util.ObjectHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPClientConfig;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPReply;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class FtpOperations
implements RemoteFileOperations<FTPFile> {
    protected final transient Log log = LogFactory.getLog(this.getClass());
    protected final FTPClient client;
    protected final FTPClientConfig clientConfig;
    protected RemoteFileEndpoint<FTPFile> endpoint;

    public FtpOperations(FTPClient client, FTPClientConfig clientConfig) {
        this.client = client;
        this.clientConfig = clientConfig;
    }

    public void setEndpoint(GenericFileEndpoint<FTPFile> endpoint) {
        this.endpoint = (RemoteFileEndpoint)endpoint;
    }

    @Override
    public boolean connect(RemoteFileConfiguration configuration) throws GenericFileOperationFailedException {
        if (this.log.isTraceEnabled()) {
            this.log.trace((Object)("Connecting using FTPClient: " + this.client));
        }
        String host = configuration.getHost();
        int port = configuration.getPort();
        String username = configuration.getUsername();
        if (this.clientConfig != null) {
            this.log.trace((Object)("Configuring FTPClient with config: " + this.clientConfig));
            this.client.configure(this.clientConfig);
        }
        if (this.log.isTraceEnabled()) {
            this.log.trace((Object)("Connecting to " + configuration.remoteServerInformation()));
        }
        boolean connected = false;
        int attempt = 0;
        while (!connected) {
            try {
                if (this.log.isTraceEnabled() && attempt > 0) {
                    this.log.trace((Object)("Reconnect attempt #" + attempt + " connecting to + " + configuration.remoteServerInformation()));
                }
                this.client.connect(host, port);
                int reply = this.client.getReplyCode();
                if (FTPReply.isPositiveCompletion((int)reply)) {
                    connected = true;
                    continue;
                }
                throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), "Server refused connection");
            }
            catch (Exception e) {
                GenericFileOperationFailedException failed = e instanceof GenericFileOperationFailedException ? (GenericFileOperationFailedException)e : new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
                if (this.log.isTraceEnabled()) {
                    this.log.trace((Object)("Cannot connect due: " + failed.getMessage()));
                }
                if (++attempt > this.endpoint.getMaximumReconnectAttempts()) {
                    throw failed;
                }
                if (this.endpoint.getReconnectDelay() <= 0L) continue;
                try {
                    Thread.sleep(this.endpoint.getReconnectDelay());
                }
                catch (InterruptedException ie) {}
            }
        }
        if (configuration.isPassiveMode()) {
            this.log.trace((Object)"Using passive mode connections");
            this.client.enterLocalPassiveMode();
        }
        try {
            boolean login;
            if (username != null) {
                if (this.log.isTraceEnabled()) {
                    this.log.trace((Object)("Attempting to login user: " + username + " using password: " + configuration.getPassword()));
                }
                login = this.client.login(username, configuration.getPassword());
            } else {
                this.log.trace((Object)"Attempting to login anonymous");
                login = this.client.login("anonymous", null);
            }
            if (this.log.isTraceEnabled()) {
                this.log.trace((Object)("User " + (username != null ? username : "anonymous") + " logged in: " + login));
            }
            if (!login) {
                throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString());
            }
            this.client.setFileType(configuration.isBinary() ? 2 : 0);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        return true;
    }

    @Override
    public boolean isConnected() throws GenericFileOperationFailedException {
        return this.client.isConnected();
    }

    @Override
    public void disconnect() throws GenericFileOperationFailedException {
        try {
            this.client.logout();
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        finally {
            try {
                this.client.disconnect();
            }
            catch (IOException e) {
                throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
            }
        }
    }

    public boolean deleteFile(String name) throws GenericFileOperationFailedException {
        if (this.log.isDebugEnabled()) {
            this.log.debug((Object)("Deleting file: " + name));
        }
        try {
            return this.client.deleteFile(name);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public boolean renameFile(String from, String to) throws GenericFileOperationFailedException {
        if (this.log.isDebugEnabled()) {
            this.log.debug((Object)("Renaming file: " + from + " to: " + to));
        }
        try {
            return this.client.rename(from, to);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean buildDirectory(String directory, boolean absolute) throws GenericFileOperationFailedException {
        boolean bl;
        block10: {
            if (this.log.isTraceEnabled()) {
                this.log.trace((Object)("Building directory: " + directory));
            }
            String originalDirectory = this.client.printWorkingDirectory();
            try {
                boolean success = this.client.changeWorkingDirectory(directory);
                if (!success) {
                    if (this.log.isTraceEnabled()) {
                        this.log.trace((Object)("Trying to build remote directory: " + directory));
                    }
                    if (!(success = this.client.makeDirectory(directory))) {
                        success = this.buildDirectoryChunks(directory);
                    }
                }
                bl = success;
                if (originalDirectory == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (originalDirectory != null) {
                        this.client.changeWorkingDirectory(originalDirectory);
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
                }
            }
            this.client.changeWorkingDirectory(originalDirectory);
        }
        return bl;
    }

    public boolean retrieveFile(String name, Exchange exchange) throws GenericFileOperationFailedException {
        if (ObjectHelper.isNotEmpty((Object)this.endpoint.getLocalWorkDirectory())) {
            return this.retrieveFileToFileInLocalWorkDirectory(name, exchange);
        }
        return this.retrieveFileToStreamInBody(name, exchange);
    }

    private boolean retrieveFileToStreamInBody(String name, Exchange exchange) throws GenericFileOperationFailedException {
        ByteArrayOutputStream os = null;
        try {
            os = new ByteArrayOutputStream();
            GenericFile target = (GenericFile)exchange.getProperty("CamelFileExchangeFile");
            ObjectHelper.notNull((Object)target, (String)"Exchange should have the CamelFileExchangeFile set");
            target.setBody((Object)os);
            boolean bl = this.client.retrieveFile(name, (OutputStream)os);
            return bl;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        finally {
            ObjectHelper.close((Closeable)os, (String)("retrieve: " + name), (Log)this.log);
        }
    }

    private boolean retrieveFileToFileInLocalWorkDirectory(String name, Exchange exchange) throws GenericFileOperationFailedException {
        boolean result;
        FileOutputStream os;
        File temp;
        File local = new File(FileUtil.normalizePath((String)this.endpoint.getLocalWorkDirectory()));
        try {
            GenericFile target = (GenericFile)exchange.getProperty("CamelFileExchangeFile");
            ObjectHelper.notNull((Object)target, (String)"Exchange should have the CamelFileExchangeFile set");
            String relativeName = target.getRelativeFilePath();
            temp = new File(local, relativeName + ".inprogress");
            local = new File(local, relativeName);
            local.mkdirs();
            if (temp.exists() && !FileUtil.deleteFile((File)temp)) {
                throw new GenericFileOperationFailedException("Cannot delete existing local work file: " + temp);
            }
            if (local.exists() && !FileUtil.deleteFile((File)local)) {
                throw new GenericFileOperationFailedException("Cannot delete existing local work file: " + local);
            }
            if (!temp.createNewFile()) {
                throw new GenericFileOperationFailedException("Cannot create new local work file: " + temp);
            }
            os = new FileOutputStream(temp);
            exchange.getIn().setHeader("CamelFileLocalWorkPath", (Object)local.getPath());
        }
        catch (Exception e) {
            throw new GenericFileOperationFailedException("Cannot create new local work file: " + local);
        }
        try {
            GenericFile target = (GenericFile)exchange.getProperty("CamelFileExchangeFile");
            target.setBody((Object)local);
            result = this.client.retrieveFile(name, (OutputStream)os);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        finally {
            ObjectHelper.close((Closeable)os, (String)("retrieve: " + name), (Log)this.log);
        }
        if (this.log.isDebugEnabled()) {
            this.log.debug((Object)("Retrieve file to local work file result: " + result));
        }
        if (result) {
            if (this.log.isTraceEnabled()) {
                this.log.trace((Object)("Renaming local in progress file from: " + temp + " to: " + local));
            }
            if (!FileUtil.renameFile((File)temp, (File)local)) {
                throw new GenericFileOperationFailedException("Cannot rename local work file from: " + temp + " to: " + local);
            }
        }
        return result;
    }

    public boolean storeFile(String name, Exchange exchange) throws GenericFileOperationFailedException {
        if (this.endpoint.getFileExist() == GenericFileExist.Ignore || this.endpoint.getFileExist() == GenericFileExist.Fail) {
            boolean existFile = this.existsFile(name);
            if (existFile && this.endpoint.getFileExist() == GenericFileExist.Ignore) {
                if (this.log.isTraceEnabled()) {
                    this.log.trace((Object)("An existing file already exists: " + name + ". Ignore and do not override it."));
                }
                return true;
            }
            if (existFile && this.endpoint.getFileExist() == GenericFileExist.Fail) {
                throw new GenericFileOperationFailedException("File already exist: " + name + ". Cannot write new file.");
            }
        }
        InputStream is = null;
        try {
            is = (InputStream)exchange.getIn().getMandatoryBody(InputStream.class);
            if (this.endpoint.getFileExist() == GenericFileExist.Append) {
                boolean bl = this.client.appendFile(name, is);
                return bl;
            }
            boolean bl = this.client.storeFile(name, is);
            return bl;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        catch (InvalidPayloadException e) {
            throw new GenericFileOperationFailedException("Cannot store file: " + name, (Throwable)e);
        }
        finally {
            ObjectHelper.close((Closeable)is, (String)("store: " + name), (Log)this.log);
        }
    }

    public boolean existsFile(String name) throws GenericFileOperationFailedException {
        String directory = FileUtil.onlyPath((String)name);
        if (directory == null) {
            return false;
        }
        String onlyName = FileUtil.stripPath((String)name);
        try {
            String[] names = this.client.listNames(directory);
            if (names == null) {
                return false;
            }
            for (String existing : names) {
                if (!existing.equals(onlyName)) continue;
                return true;
            }
            return false;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public String getCurrentDirectory() throws GenericFileOperationFailedException {
        try {
            return this.client.printWorkingDirectory();
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public void changeCurrentDirectory(String newDirectory) throws GenericFileOperationFailedException {
        try {
            this.client.changeWorkingDirectory(newDirectory);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public List<FTPFile> listFiles() throws GenericFileOperationFailedException {
        return this.listFiles(".");
    }

    public List<FTPFile> listFiles(String path) throws GenericFileOperationFailedException {
        if (ObjectHelper.isEmpty((Object)path)) {
            path = ".";
        }
        try {
            ArrayList<FTPFile> list = new ArrayList<FTPFile>();
            FTPFile[] files = this.client.listFiles(path);
            if (files != null) {
                list.addAll(Arrays.asList(files));
            }
            return list;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    @Override
    public boolean sendNoop() throws GenericFileOperationFailedException {
        try {
            return this.client.sendNoOp();
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    private boolean buildDirectoryChunks(String dirName) throws IOException {
        StringBuilder sb = new StringBuilder(dirName.length());
        String[] dirs = dirName.split("/|\\\\");
        boolean success = false;
        for (String dir : dirs) {
            sb.append(dir).append('/');
            String directory = sb.toString();
            if (directory.equals("/")) continue;
            if (this.log.isTraceEnabled()) {
                this.log.trace((Object)("Trying to build remote directory by chunk: " + directory));
            }
            success = this.client.makeDirectory(directory);
        }
        return success;
    }
}

