/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MessageFormatter.java 19784 2010-04-28 17:05:13Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.component;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.camel.Exchange;
import org.apache.camel.RouteNode;
import org.apache.camel.model.ProcessorDefinition;
import org.apache.camel.processor.interceptor.TraceFormatter;
import org.apache.camel.processor.interceptor.TraceInterceptor;
import org.apache.camel.spi.TracedRouteNodes;

public class MessageFormatter implements TraceFormatter {

    private static Logger logger = Logger.getLogger(MessageFormatter.class.getName());

    @SuppressWarnings("unchecked")
    public Object format(final TraceInterceptor interceptor, final ProcessorDefinition node, final Exchange exchange) {
        return MessageFormatter.formatMessage(exchange);
    }

    public static String formatMessage(final Exchange exchange) {
        String id = exchange.getExchangeId();

        // compute from and to
        String from = "";
        String to = "";

        TracedRouteNodes tracedNode = exchange.getUnitOfWork().getTracedRouteNodes();
        RouteNode traceFrom = tracedNode.getSecondLastNode();
        if (traceFrom != null) {
            from = traceFrom.getLabel(exchange);
        } else if (exchange.getFromEndpoint() != null) {
            from = exchange.getFromEndpoint().getEndpointUri();
        }

        RouteNode traceTo = tracedNode.getLastNode();
        if (traceTo != null) {
            to = traceTo.getLabel(exchange);
        }

        Throwable exception = exchange.getProperty(Exchange.EXCEPTION_CAUGHT, Throwable.class);

        String filename = null;
        if (exchange.hasOut()) {
            filename = exchange.getOut().getHeader(Exchange.FILE_NAME, String.class);
        }
        if (filename == null) {
            filename = exchange.getIn().getHeader(Exchange.FILE_NAME, String.class);
        }
        if (filename == null) {
            // No file name, for example an incoming CXF call
            filename = "";
        }

        String filenameAbsolutePath = null;
        if (exchange.hasOut()) {
            filenameAbsolutePath = exchange.getOut().getHeader("CamelFileAbsolutePath", String.class);
        }
        if (filenameAbsolutePath == null) {
            filenameAbsolutePath = exchange.getIn().getHeader("CamelFileAbsolutePath", String.class);
        }
        if (filenameAbsolutePath == null) {
            // No file name, for example an incoming CXF call
            filenameAbsolutePath = "";
        }

        return MessageFormatter.formatMessage(from, to, id, exception, filename, filenameAbsolutePath);
    }

    public static String formatMessage(final String from, final String to, final String exchangeId, final Throwable exception,
        final String filename, final String filenameAbsolutePath) {

        String filenameAbsoluteWithSlash = filenameAbsolutePath;
        if (filenameAbsoluteWithSlash.contains("\\")) {
            filenameAbsoluteWithSlash = filenameAbsoluteWithSlash.replace("\\", "/");
        }
        if (!filenameAbsoluteWithSlash.startsWith("/")) {
            filenameAbsoluteWithSlash = "/" + filenameAbsoluteWithSlash;
        }

        Map<String, Object> headers = new HashMap<String, Object>();
        if (exchangeId != null) {
            headers.put("MessageId", exchangeId);
        }
        headers.put("From", MessageFormatter.filterParameters(from));
        headers.put("To", MessageFormatter.filterParameters(to));
        headers.put("FileName", filename);
        headers.put("FileNameAbsolute", filenameAbsoluteWithSlash);

        if (exception != null) {
            headers.put("HasException", true);
            headers.put("ExceptionClass", exception.getClass().getName());
            if (exception.getMessage() == null) {
                headers.put("ExceptionMessage", "");
            } else {
                try {
                    headers.put("ExceptionMessage", URLEncoder.encode(exception.getMessage(), "UTF-8"));
                } catch (UnsupportedEncodingException e) {
                    // This should never happen, UTF-8 shall always exist
                    MessageFormatter.logger.log(Level.SEVERE, "Cannot format message", e);
                    headers.put("ExceptionMessage", "");
                }
            }
        } else {
            headers.put("HasException", false);
        }

        return headers.toString();
    }

    public static String filterParameters(final String url) {
        if (url == null) {
            return null;
        } else {
            final int questionMark = url.indexOf('?');
            if (questionMark == -1) {
                return url;
            } else {
                return url.substring(0, questionMark);
            }
        }
    }

}
