/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: RegistryWrapper.java 19913 2010-06-02 16:00:27Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.registry.xml;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import org.apache.camel.CamelContext;
import org.apache.camel.impl.DefaultCamelContext;
import org.apache.camel.impl.SimpleRegistry;
import org.apache.camel.osgi.CompositeRegistry;
import org.apache.camel.spi.Registry;
import org.ow2.jonas.camel.registry.xml.generated.Entry;
import org.ow2.jonas.camel.registry.xml.loader.XmlLoader;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Wraps a {@link Registry} for a given {@link CamelContext}. A new
 * {@link SimpleRegistry} is aggregated to the current registry of the context.
 * This new resgistry will store all the entries parsed from xml files.<br>
 * <br>
 * Here is an example of the registry file:<br>
 * <ul>
 * <nobr> <code>
 * &lt;registry xmlns="http://jonas.ow2.org/camel-file-registry-1.0"<br>
 * &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"<br>
 * &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; xsi:schemaLocation="http://jonas.ow2.org/camel-file-registry-1.0<br>
 * &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; http://jonas.ow2.org/ns/camel-registry/file-registry-1.0.xsd"&gt;<br>
 * <br>
 * &nbsp; &lt;entry&gt;<br>
 * &nbsp; &nbsp; &lt;logicalName&gt;queue&lt;/logicalName&gt;<br>
 * &nbsp; &nbsp; &lt;technicalName&gt;test-jms:queue:sampleQueue&lt;/technicalName&gt;<br>
 * &nbsp; &lt;/entry&gt;<br>
 * <br>
 * &nbsp; &lt;entry&gt;<br>
 * &nbsp; &nbsp; &lt;logicalName&gt;multiline-with-spaces&lt;/logicalName&gt;<br>
 * &nbsp; &nbsp; &lt;technicalName&gt;<br>
 * &nbsp; &nbsp; &nbsp; file://C:<br>
 * &nbsp; &nbsp; &nbsp; /Documents and Settings/<br>
 * &nbsp; &nbsp; &nbsp; My User<br>
 * &nbsp; &nbsp; &lt;/technicalName&gt;<br>
 * &nbsp; &lt;/entry&gt;<br>
 * &lt;/registry>
 * </code> </nobr>
 * </ul>
 * 
 * @see <a href="http://jonas.ow2.org/ns/camel-registry/file-registry-1.0.xsd">
 *      http://jonas.ow2.org/ns/camel-registry/file-registry-1.0.xsd</a> for the
 *      XSD schema of the registry XML file.
 * @author eyindanga
 */
public class RegistryWrapper {

    private static final String XSD_URL = "org/ow2/jonas/camel/registry/xml/xsd/file-registry-1.0.xsd";

    /**
     * Gets the {@link SimpleRegistry} containing entries parsed from xml files.
     * 
     * @return the registry
     */
    public SimpleRegistry getRegistry() {
        return this.registry;
    }

    /**
     * Sets the registry that will store entries parsed from xml files.
     * 
     * @param registry the registry to set
     */
    public void setRegistry(final SimpleRegistry registry) {
        this.registry = registry;
    }

    /**
     * The logger.
     */
    private final Log logger = LogFactory.getLog(this.getClass());

    /**
     * The simple registry.
     */
    private SimpleRegistry registry;

    /**
     * Camel context.
     */

    private CamelContext ctxt = null;

    /**
     * Public constructor.
     */
    public RegistryWrapper() {
    }

    /**
     * Creates a {@link CompositeRegistry} containing the current context of the
     * registry , added in first position, plus a {@link SimpleRegistry} where
     * all entries parsed from xml files will be stored.
     * 
     * @param ctxt the {@link CamelContext}
     * @return a <code>CompositeRegistry</code>.
     */
    public CompositeRegistry createRegistry(final CamelContext ctxt) {
        this.ctxt = ctxt;
        Registry reg = this.ctxt.getRegistry();
        CompositeRegistry cr = null;
        if (!(reg instanceof CompositeRegistry) || this.registry == null) {
            if (this.registry == null) {
                this.registry = new SimpleRegistry();
            }
            ArrayList<Registry> listRepo = new ArrayList<Registry>(2);
            listRepo.add(reg);
            listRepo.add(this.registry);
            cr = new CompositeRegistry(listRepo);
        }
        return cr;
    }

    /**
     * Add entries parsed from an XML file into the wrapped registry.
     * 
     * @see {@link RegistryWrapper} for the XML file example.
     * @param input The stream of the registry file that will be parsed to get
     *        registry entries.
     * @throws Exception any.
     */
    public void addValues(final InputStream input) throws Exception {
        try {
            org.ow2.jonas.camel.registry.xml.generated.Registry registry = XmlLoader.loadSchemaAndFile(RegistryWrapper.XSD_URL,
                org.ow2.jonas.camel.registry.xml.generated.Registry.class, input);

            for (Entry entry : registry.getEntry()) {
                final String name = this.removeAllWhitespace(entry.getLogicalName());
                final String value = this.removeAllWhitespace(entry.getTechnicalName());

                if (this.registry.containsKey(name)) {
                    throw new IllegalStateException("The entry name " + name + " is already set.");
                }
                this.registry.put(name, this.ctxt.getEndpoint(value));
            }
        } finally {
            try {
                input.close();
            } catch (IOException e) {
                this.logger.error("Error while closing the inputStream", e);
            }
        }
    }

    /**
     * Removes entries from the registry component.
     * 
     * @param input the stream of the registry file that will be parsed to get
     *        registry entries.
     * @throws IllegalArgumentException if removal of any entry fails
     */
    private void removeValues(final InputStream input) throws IllegalArgumentException {
        try {
            org.ow2.jonas.camel.registry.xml.generated.Registry registry = XmlLoader.loadSchemaAndFile(RegistryWrapper.XSD_URL,
                org.ow2.jonas.camel.registry.xml.generated.Registry.class, input);

            for (Entry entry : registry.getEntry()) {
                final String name = this.removeAllWhitespace(entry.getLogicalName());

                if (this.registry.containsKey(name)) {
                    this.registry.remove(name);
                } else {
                    this.logger.warn("The entry {0} does not exist in the registry.", name);
                }
            }
        } catch (Throwable t) {
            throw new IllegalArgumentException(t.getMessage(), t);
        } finally {
            try {
                input.close();
            } catch (IOException e) {
                this.logger.error("Error while closing the inputStream", e);
            }
        }
    }

    /**
     * Add entries parsed from xml files into the wrapped registry.
     * 
     * @see {@link RegistryWrapper} for the XML file example.
     * @param input The stream of the registry file that will be parsed to get
     *        registry entries.
     * @throws Exception any.
     */
    public void addToTheRegistry(final InputStream inputStream) throws Exception {
        this.addValues(inputStream);
    }

    /**
     * Removes entries from the registry component.
     * 
     * @see {@link RegistryWrapper} for the XML file example.
     * @param input the stream of the registry file that will be parsed to get
     *        registry entries.
     * @throws IllegalArgumentException if removal of any entry fails
     */
    public void removeFromTheRegistry(final InputStream inputStream) throws IllegalArgumentException {
        this.removeValues(inputStream);
    }

    /**
     * Removes all white spaces from a string.
     * 
     * @param input Input.
     * @return the string value without white spaces.
     */
    private String removeAllWhitespace(String input) {
        StringBuilder result = new StringBuilder();
        input = input.replace('\r', '\n');
        for (String line : input.split("\n")) {
            result.append(line.trim());
        }
        return result.toString();
    }

    /**
     * Set the current Camel context
     * 
     * @param ctxt the context to set.
     */
    public void createRegistryAndAssignToContext(final DefaultCamelContext ctxt) {
        Registry reg = ctxt.getRegistry();
        if (!(reg instanceof CompositeRegistry) || this.registry == null) {
            if (this.registry == null) {
                this.registry = new SimpleRegistry();
            }
            ArrayList<Registry> listRepo = new ArrayList<Registry>(2);
            listRepo.add(reg);
            listRepo.add(this.registry);
            CompositeRegistry cr = new CompositeRegistry(listRepo);
            ctxt.setRegistry(cr);
        }
        this.ctxt = ctxt;
    }

}
