/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelService.java 19913 2010-06-02 16:00:27Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.service.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.camel.CamelContext;
import org.apache.camel.Component;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.impl.SimpleRegistry;
import org.apache.camel.model.RouteDefinition;
import org.apache.camel.osgi.CamelContextFactory;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceFactory;
import org.osgi.framework.ServiceRegistration;
import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.jonas.camel.wrapper.CamelWrapper;
import org.ow2.jonas.camel.wrapper.ICamelWrapper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Camel service.
 * 
 * @author Guillaume Renault
 */
public class CamelService implements ICamelService {

    /**
     * Logger.
     */
    private final Log logger = LogFactory.getLog(CamelService.class);

    /**
     * Camel context factory.
     */
    private CamelContextFactory camelFactory = null;

    /**
     * Map of all of the camel contexts.
     */
    private Map<String, ICamelWrapper> camelContexts = null;

    /**
     * the camel context {@link ServiceFactory}.
     */
    private ServiceRegistration camelContextFactoryServiceRegistration;

    private BundleContext bundleContext;

    /**
     * Constructor, needed to get the bundle context.
     * 
     * @param context the bundle context.
     */
    public CamelService(final BundleContext context) {
        this.camelFactory = new CamelContextFactory();
        this.camelFactory.setBundleContext(context);
        this.bundleContext = context;
    }

    /**
     * Initializes the service.
     */
    public void start() {
        this.camelContexts = new HashMap<String, ICamelWrapper>();

        this.logger.info("Camel service started");

        if (this.bundleContext != null) {
            CamelContextServiceFactory camelContextServiceFactory = new CamelContextServiceFactory();
            camelContextServiceFactory.setCamelService(this);
            this.camelContextFactoryServiceRegistration = this.bundleContext.registerService(CamelContext.class.getName(),
                camelContextServiceFactory, null);

            this.logger.info("Camel context factory service started");
        }
    }

    /**
     * Stops all Camel contexts.
     */
    public void stop() {
        if (this.camelContextFactoryServiceRegistration != null) {
            this.camelContextFactoryServiceRegistration.unregister();
            this.logger.info("Camel context factory service stopped");
        }
        final List<String> stoppedContexts = new ArrayList<String>();
        final Map<String, ICamelWrapper> camelContexts = new HashMap<String, ICamelWrapper>(this.camelContexts);
        this.camelContexts = new HashMap<String, ICamelWrapper>();
        for (final Map.Entry<String, ICamelWrapper> entry : camelContexts.entrySet()) {
            try {
                entry.getValue().stopContext();
                stoppedContexts.add(entry.getKey());
                this.logger.debug("The camel context {0} has been stopped.", entry.getKey());
            } catch (final Exception e) {
                this.logger.warn("Cannot stop the camel context {0}.", entry.getKey(), e);
            }
        }
        for (final String stoppedContext : stoppedContexts) {
            camelContexts.remove(stoppedContext);
        }

        if (camelContexts.size() != 0) {
            this.logger.warn("The Camel service still has {0} running contexts!", camelContexts.size());
        }

        this.logger.info("Camel service stopped");
    }

    /**
     * Get the names of Camel contexts managed by the service.
     * 
     * @return Names of Camel contexts managed by the service.
     */
    public List<String> getContextNames() {
        return new ArrayList<String>(this.camelContexts.keySet());
    }

    /**
     * Start a new camel context.
     * 
     * @return the name of the camel context.
     */
    public String startNewContext() throws Exception {
        this.logger.info("Starting a new camel context");

        final CamelWrapper camelWrapper = new CamelWrapper(this.camelFactory);

        // put the Camel wrapper in the wrappers list.
        this.camelContexts.put(camelWrapper.getCamelContextName(), camelWrapper);

        // set the wrapper's name
        camelWrapper.setName(camelWrapper.getCamelContextName());

        // start the context.
        camelWrapper.run();

        this.logger.debug("A new Camel context has been created with the name {0}", camelWrapper.getCamelContextName());

        return camelWrapper.getCamelContextName();
    }

    /**
     * Stop an instance of Camel.
     * 
     * @param name the camel context name to stop.
     * @throws Exception If stopping any CAMEL console fails.
     */
    public void stop(final String name) throws Exception {
        if (this.camelContexts.containsKey(name)) {
            this.camelContexts.get(name).stopContext();
            this.camelContexts.remove(name);

            this.logger.debug("The camel context {0} has been stopped.", name);
        }
    }

    /**
     * Get an instance of Camel.
     * 
     * @param name the camel context name
     * @return the camel context
     */
    public CamelContext getCamelContext(final String name) {
        if (this.camelContexts.containsKey(name)) {
            final ICamelWrapper camelWrapper = this.camelContexts.get(name);
            return camelWrapper.getCamelContext();
        }
        return null;
    }

    /**
     * Get a simple registry instance of Camel context.
     * 
     * @param name the camel context name
     * @return the registry instance
     */
    public SimpleRegistry getSimpleRegistryOfCamelContext(final String name) {
        if (this.camelContexts.containsKey(name)) {
            final ICamelWrapper camelWrapper = this.camelContexts.get(name);
            return camelWrapper.getRegistryWrapper().getRegistry();
        }
        return null;
    }

    /**
     * Start a new camel context and return context.
     * 
     * @return the camel context.
     * @throws Exception If starting the context fails.
     */
    public CamelContext startAndGetNewContext() throws Exception {
        final String contextName = this.startNewContext();
        return this.getCamelContext(contextName);
    }

    /**
     * @param builder the route to add.
     * @param camelContextName the camel context name on which the route has to
     *        be set.
     * @throws Exception If adding route fails.
     */
    public void addRoutes(final RouteBuilder builder, final String camelContextName) throws Exception {
        this.camelContexts.get(camelContextName).addRoutes(builder);
        this.logger.debug("A route has been added on the following Camel context : {0}", camelContextName);
    }

    /**
     * Get the list of existing routes.
     * 
     * @param camelContextName the identifier of the camel context to check.
     * @return the list of existing routes for the given camel context..
     */
    public List<RouteDefinition> getRouteDefinitions(final String camelContextName) {
        return this.camelContexts.get(camelContextName).getRouteDefinitions();
    }

    /**
     * Get a producer template for the given camel context.
     * 
     * @param camelContextName the camel context name on which the producer has
     *        to be taken.
     * @return the producer template.
     */
    public ProducerTemplate getProducerTemplate(final String camelContextName) {
        this.logger.debug("Create a new producer template on the following Camel context : {0}", camelContextName);
        return this.camelContexts.get(camelContextName).getProducerTemplate();
    }

    /**
     * Add values to the registry component.
     * 
     * @param input the stream that contains the entries to add.
     * @param camelContextName the camel context name that contains the registry
     *        to update.
     */
    public void addRegistry(final InputStream input, final String camelContextName) throws Exception {
        this.logger.debug("Add entries to the Registry component of the following Camel context : {0}", camelContextName);
        this.camelContexts.get(camelContextName).addToTheRegistry(input);
    }

    /**
     * Remove entries from the registry component.
     * 
     * @param input the input stream that contains all the entries to remove
     * @param camelContextName the camel context name to remove the entries on.
     */
    public void removeRegistry(final InputStream input, final String camelContextName) {
        this.camelContexts.get(camelContextName).removeFromTheRegistry(input);
        this.logger
            .debug("Some entries were removed from the registry of the following camel context : {0}.", camelContextName);
    }

    /**
     * Add a component on the given camel context.
     * 
     * @param componentName the component name.
     * @param component the component.
     * @param camelContextName the camel name context.
     */
    public void addComponent(final String componentName, final Component component, final String camelContextName) {
        // register the component on the camel context.
        this.camelContexts.get(camelContextName).addComponent(componentName, component);

        this.logger.debug("Component {0} added on the Camel context {1}", componentName, camelContextName);
    }

}
