/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009-2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleJMS.java 19967 2010-06-09 13:47:02Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.example.jms;

import java.io.File;

import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.osgi.framework.BundleContext;
import org.ow2.jonas.camel.component.RouteBuilderComponent;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class ExampleJMS extends RouteBuilderComponent {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The JONAS_BASE/logs folder.
     */
    private File jonasLogs;

    /**
     * Constructor: saves the BundleContext.
     */
    public ExampleJMS(final BundleContext bundleContext) {
        super(bundleContext);
    }

    /**
     * Saves the <code>$JONAS_BASE/logs</code> directory, calls
     * {@link RouteBuilderComponent#start()} and launches a self-test.
     */
    @Override
    public void start() throws Throwable {
        String jonasBase = System.getProperty("jonas.base");
        if (jonasBase == null) {
            throw new IllegalStateException("jonas.base is not set!");
        }
        File jonasBaseFile = new File(jonasBase);
        if (!jonasBaseFile.isDirectory()) {
            throw new IllegalStateException(jonasBaseFile + " is not a directory");
        }
        this.jonasLogs = new File(jonasBaseFile, "logs");
        if (!this.jonasLogs.isDirectory()) {
            throw new IllegalStateException(this.jonasLogs + " is not a directory");
        }

        super.start();

        // For the internal test, we need to change the class loader
        //
        // Note that in the CAMEL routes, the class loader is always correct so
        // we don't need to worry about this...
        final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            this.test();
        } finally {
            Thread.currentThread().setContextClassLoader(oldCL);
        }
    }

    /**
     * Add routes to the CAMEL context.
     */
    @Override
    public void configure() throws Exception {
        this.from("registry:queue").process(new Processor() {
            public void process(final Exchange e) throws Exception {
                String message = e.getIn().getBody(String.class);
                ExampleJMS.this.logger.info("Received JMS message: " + message);
                File log = new File(ExampleJMS.this.jonasLogs, message);
                log.createNewFile();
            }
        });
    }

    /**
     * Test the route.
     */
    public void test() throws Exception {
        Context context = new InitialContext();
        QueueConnectionFactory connectionFactory = (QueueConnectionFactory) context.lookup("JQCF");
        QueueConnection queueConnection = connectionFactory.createQueueConnection();
        QueueSession queueSession = queueConnection.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
        Queue queue = (Queue) context.lookup("sampleQueue");
        QueueSender jmsSender = queueSession.createSender(queue);

        TextMessage message = queueSession.createTextMessage("Test Message");
        jmsSender.send(message);
    }

}
