/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XmlLoader.java 19967 2010-06-09 13:47:02Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.registry.xml.loader;

import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Unmarshaller;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Loads an XML file using JAXB.
 */
public class XmlLoader {

    private static Log logger = LogFactory.getLog(XmlLoader.class);

    /**
     * Unmarshallers cache.
     */
    private static Map<String, Unmarshaller> unMarshallers = new HashMap<String, Unmarshaller>();

    /**
     * Verifies an XML file against an XSD and instantiates it using a given
     * class.
     * 
     * @param xsdPath XSD file path.
     * @param xmlRoot Root element of the XML
     * @param xmlInputStream XML to load.
     * @return XML loaded using JAXB and the rootClass.
     * @throws Exception If creating the schema or parsing the XML fails.
     */
    @SuppressWarnings("unchecked")
    public static <T> T loadSchemaAndFile(final String xsdPath, final Class<T> rootClass, final InputStream xmlInputStream)
        throws Exception {

        final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(rootClass.getClassLoader());

            Unmarshaller unMarshaller = XmlLoader.getUnMarshaller(xsdPath, rootClass);

            XmlLoader.logger.debug("Unmarshalling XML");
            return (T) (((JAXBElement) unMarshaller.unmarshal(xmlInputStream)).getValue());
        } finally {
            Thread.currentThread().setContextClassLoader(oldCL);
            XmlLoader.logger.debug("Exiting loadSchemaAndFile");
        }
    }

    private static synchronized Unmarshaller getUnMarshaller(final String xsdPath, final Class<?> rootClass) throws Exception {
        final String identifier = rootClass.getName() + "#" + xsdPath;
        Unmarshaller unMarshaller = XmlLoader.unMarshallers.get(identifier);
        if (unMarshaller == null) {
            XmlLoader.logger.debug("Creating new JAXBContext for {0}", rootClass.getPackage().getName());
            JAXBContext jc = JAXBContext.newInstance(rootClass.getPackage().getName());
            unMarshaller = jc.createUnmarshaller();

            XmlLoader.logger.debug("Setting unMarshaller schema");
            URL xsdURL = rootClass.getClassLoader().getResource(xsdPath);
            SchemaFactory schemaFactory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
            Schema schema = schemaFactory.newSchema(xsdURL);
            unMarshaller.setSchema(schema);

            XmlLoader.logger.debug("Putting into cache idenfitier {0}, unMarshaller {1}", identifier, unMarshaller);
            XmlLoader.unMarshallers.put(identifier, unMarshaller);
        }
        return unMarshaller;
    }
}
