/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: OSGiRegistry.java 19445 2010-03-15 16:46:20Z porcherg $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.registry.osgi;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.camel.spi.Registry;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of {@link Registry} for looking OSGi services.
 * 
 * @author Guillaume Porcher
 */
public class OSGiRegistry implements Registry {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * the bundle context.
     */
    private final BundleContext bundleContext;

    /**
     * Default constructor.
     * 
     * @param bundleContext the current bundle context
     */
    public OSGiRegistry(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    /**
     * Looks for an OSGi service exporting a name property equals to "name".
     * 
     * @see org.apache.camel.spi.Registry#lookup(String)
     */
    public Object lookup(final String name) {
        this.logger.debug("Looking up name {0}", name);
        return this.lookup(name, null);
    }

    /**
     * Looks for an OSGi service exporting a name property equals to "name" and
     * with class clazz.
     * 
     * @see org.apache.camel.spi.Registry#lookup(java.lang.String,
     *      java.lang.Class)
     */
    public <T> T lookup(final String name, final Class<T> clazz) {
        this.logger.debug("Looking up name {0}, class {1}", name, clazz);
        return this.lookupByType(clazz).get(name);
    }

    /**
     * Looks for an OSGi service with class clazz.
     * 
     * @see org.apache.camel.spi.Registry#lookupByType(java.lang.Class)
     */
    @SuppressWarnings("unchecked")
    public <T> Map<String, T> lookupByType(final Class<T> clazz) {
        this.logger.debug("Looking up class {0}", clazz);

        if (this.bundleContext != null) {
            try {
                ServiceReference[] refs = this.bundleContext.getServiceReferences(null, null);
                if (refs != null && refs.length >= 0) {
                    Map<String, T> result = new HashMap<String, T>();

                    for (ServiceReference ref : refs) {
                        String serviceName = (String) ref.getProperty("name");
                        if (serviceName != null) {
                            Object service = this.bundleContext.getService(ref);
                            if (service != null) {
                                this.logger.debug("\tChecking service named {0} with class {1} against class {2}", serviceName,
                                    service.getClass(), clazz);
                                if (clazz == null || clazz.isAssignableFrom(service.getClass())) {
                                    result.put(serviceName, (T) service);
                                }
                            }
                            // TODO: unget service when context is stopped.
                            // this.bundleContext.ungetService(ref);
                        }
                    }

                    this.logger.debug("Looked up for class {0}, returning {1}", clazz, result);
                    return result;
                }
            } catch (InvalidSyntaxException e) {
                this.logger.error("Failed listing OSGi services", e);
            }
        }
        this.logger.debug("Looked up for class {0}, returning empty map", clazz);
        return Collections.EMPTY_MAP;
    }

}
