/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelService.java 18954 2009-11-12 19:31:31Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.jonas_service;

import java.util.List;

import javax.management.ObjectName;

import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.jonas.jmx.JmxService;
import org.ow2.jonas.lib.service.AbsServiceImpl;

/**
 * Camel service. It doesn't do much, this is actually just an empty service so
 * that we can add <code>camel</code> to the list of JOnAS services.
 * 
 * @author Guillaume Renault
 */
public class CamelService extends AbsServiceImpl implements CamelServiceMBean {

    /**
     * Reference to an MBean server. Injected by iPOJO.
     */
    private JmxService jmxService = null;

    /**
     * Reference to the Camel OSGi service. Injected by iPOJO.
     */
    private ICamelService camelService = null;

    /**
     * @param jmxService JMX service to set.
     */
    public void setJmxService(final JmxService jmxService) {
        this.jmxService = jmxService;
    }

    /**
     * @param camelService Camel OSGi service to set.
     */
    public void setCamelService(final ICamelService camelService) {
        this.camelService = camelService;
    }

    /**
     * Initializes the service.
     */
    @Override
    protected void doStart() {
        this.jmxService.registerMBean(this, CamelService.getObjectName(this.getDomainName()));
    }

    /**
     * Stops all Camel contexts.
     */
    @Override
    protected void doStop() {
        this.jmxService.unregisterMBean(CamelService.getObjectName(this.getDomainName()));
    }

    /**
     * Get the names of Camel contexts managed by the underlying service.
     * 
     * @return Names of Camel contexts managed by the underlying service.
     */
    public List<String> getCamelContextNames() {
        return this.camelService.getContextNames();
    }

    /**
     * @param domainName Server's domain name.
     * @return JMX name of the versioning service management bean.
     */
    public static ObjectName getObjectName(final String domainName) {
        try {
            return ObjectName.getInstance(domainName + ":type=service,name=camel");
        } catch (Exception e) {
            throw new IllegalStateException("Cannot get camel service", e);
        }
    }

}
