/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ICamelService.java 20670 2011-01-03 17:18:43Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.service.api;

import java.io.InputStream;
import java.util.List;

import org.apache.camel.CamelContext;
import org.apache.camel.Component;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.Route;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.model.RouteDefinition;
import org.osgi.framework.BundleContext;

/**
 * Interface of the Camel Service.
 * 
 * @author Guillaume Renault
 */
public interface ICamelService {

    /**
     * Get the names of Camel contexts managed by the service.
     * 
     * @return Names of Camel contexts managed by the service.
     */
    List<String> getContextNames();

    /**
     * Start a new camel context using a given OSGi BundleContext and return
     * camel context name.
     * 
     * @param bundleContext OSGi bundle context to use.
     * @return the camel context name.
     * @throws Exception If starting the context fails.
     */
    String startNewContext(BundleContext bundleContext) throws Exception;

    /**
     * Start a new camel context using a given OSGi BundleContext and return
     * context.
     * 
     * @param bundleContext OSGi bundle context to use.
     * @return the camel context.
     * @throws Exception If starting the context fails.
     */
    CamelContext startAndGetNewContext(BundleContext bundleContext) throws Exception;

    /**
     * Stop an instance of Camel.
     * 
     * @param name the camel context name to stop.
     * @throws Exception If stopping any CAMEL console fails.
     */
    void stop(final String name) throws Exception;

    /**
     * Get an instance of Camel.
     * 
     * @param name the camel context name
     * @return the camel context
     */
    CamelContext getCamelContext(String name);

    /**
     * Add a {@link Route} to the given context, by providing a
     * {@link RouteBuilder}
     * 
     * @param builder the route builder.
     * @param camelContextName the camel context name on which the route has to
     *        be set.
     * @throws Exception If adding route fails.
     */
    void addRoutes(final RouteBuilder builder, final String camelContextName) throws Exception;

    /**
     * Get the list of existing routes.
     * 
     * @param camelContextName the identifier of the camel context to check.
     * @return the list of existing routes for the given camel context..
     */
    List<RouteDefinition> getRouteDefinitions(final String camelContextName);

    /**
     * Get a producer template for the given camel context.
     * 
     * @param camelContextName the camel context name on which the producer has
     *        to be taken.
     * @return the producer template.
     */
    ProducerTemplate getProducerTemplate(final String camelContextName);

    /**
     * Add values to the registry component.
     * 
     * @param input the stream that contains the entries to add.
     * @param camelContextName the camel context name that contains the registry
     *        to update.
     */
    void addRegistry(final InputStream input, final String camelContextName) throws Exception;

    /**
     * Remove entries from the registry component.
     * 
     * @param input the input stream that contains all the entries to remove
     * @param camelContextName the camel context name to remove the entries on.
     */
    void removeRegistry(final InputStream input, final String camelContextName);

    /**
     * Add a component on the given camel context.
     * 
     * @param componentName the component name.
     * @param component the component.
     * @param camelContextName the camel name context.
     */
    void addComponent(final String componentName, final Component component, final String camelContextName);

}
