/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelServlet.java 21288 2011-05-19 21:55:46Z renaultgu $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.component;

import org.apache.camel.CamelContext;
import org.apache.camel.Exchange;
import org.apache.camel.Producer;

import javax.servlet.*;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.HttpHeaders;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;

/**
 * Generic CAMEL servlet, that can be for example used for REST endpoints. It
 * will map all request parameters to the Camel exchange's in message headers.
 * Note that all servlet headers use {@link ServletRequest.getParameterMap()},
 * therefore the keys in the parameter map are of type String and the values in
 * the parameter map are of type String array.<br/>
 * <br/>
 * In order to use it:
 * <ul>
 * <li>Decide of a servlet path (for example, <code>/myServlet</code>) and an
 * endpoint name (for example, <code>direct:myServlet</code>).
 * <li>Register the servlet:
 * <code><pre> import org.osgi.service.http.HttpService;
 * <p/>
 * ...
 * <p/>
 * HttpService httpService = getService(HttpService.class);
 * HttpContext httpContext = httpService.createDefaultHttpContext();
 * Dictionary<String, String> initParams = new Hashtable<String, String>();
 * initParams.put("matchOnUriPrefix", "false");
 * initParams.put("servlet-name", SERVLET_PATH);
 * httpService.registerServlet(SERVLET_PATH,
 *     new CamelServlet(this.getContext(), ENDPOINT_NAME),
 *     initParams, httpContext);</pre></code></li>
 * <li>Handle incoming messages, exactly as you would do with any other CAMEL
 * route:<br/>
 * <code>this.from(ENDPOINT_NAME).process(new MyProcessor());</code></li>
 * </ul>
 */
public class CamelServlet implements Servlet {

    private CamelContext camelContext;

    private String camelDestination;

    private ServletConfig servletConfig;

    private String defaultFileName;

    /**
     * Error status code ID.
     */
    public static final String ERROR_STATUS_CODE = "javax.servlet.error.status_code";

    /**
     * Request property name for the exchange message.
     */
    public static final String REQUEST_PROPERTY_NAME = "REQUEST property";

    /**
     * Creates a new CamelServlet instance.
     *
     * @param camelContext     CAMEL context to send messages to.
     * @param camelDestination Name of the CAMEL endpoint on the camelContext
     * @param defaultFileName  Default file name to use if the response has no
     *                         file name.
     */
    public CamelServlet(final CamelContext camelContext, final String camelDestination, final String defaultFileName) {
        this.camelContext = camelContext;
        this.camelDestination = camelDestination;
        this.defaultFileName = defaultFileName;
    }

    /**
     * Creates a new CamelServlet instance.
     *
     * @param camelContext     CAMEL context to send messages to.
     * @param camelDestination Name of the CAMEL endpoint on the camelContext
     */
    public CamelServlet(final CamelContext camelContext, final String camelDestination) {
        this(camelContext, camelDestination, null);
    }

    public void init(final ServletConfig servletConfig) throws ServletException {
        this.servletConfig = servletConfig;
    }

    public ServletConfig getServletConfig() {
        return this.servletConfig;
    }

    @SuppressWarnings({"deprecation", "finally", "unchecked"})
    public void service(final ServletRequest request, final ServletResponse rsp) throws ServletException, IOException {
        final ClassLoader oldTCCL = Thread.currentThread().getContextClassLoader();

        try {
            // Change the TCCL in order to call, for example, CXF endpoints
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());

            HttpServletResponse response = (HttpServletResponse) rsp;

            Producer producer = this.camelContext.getEndpoint(this.camelDestination).createProducer();
            Exchange exchange = producer.createExchange();
            exchange.getIn().getHeaders().putAll(request.getParameterMap());
            if (null != request.getContentType()) {
                exchange.getIn().setHeader(HttpHeaders.CONTENT_TYPE, request.getContentType());
            }
            exchange.getIn().setHeader(HttpHeaders.CONTENT_LENGTH, request.getContentLength());
            exchange.getIn().setBody(request.getInputStream());
            exchange.setProperty(CamelServlet.REQUEST_PROPERTY_NAME, request);
            producer.process(exchange);

            // check for HTTP headers
            this.checkHeaders(exchange, response);

            Throwable exception = exchange.getProperty(Exchange.EXCEPTION_CAUGHT, Throwable.class);
            if (exception != null) {
                throw exception;
            }

            // Extract file name if exists
            String filename = exchange.getOut().getHeader(Exchange.FILE_NAME, String.class);
            if (filename == null) {
                filename = exchange.getIn().getHeader(Exchange.FILE_NAME, String.class);
            }
            if (filename == null) {
                filename = this.defaultFileName;
            }
            if (filename != null) {
                filename = URLEncoder.encode(filename);
                response.setHeader("Content-Type", "application/force-download; name=\"" + filename + "\"");
                response.setHeader("Content-Transfer-Encoding", "binary");
                response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
            }

            if (exchange.getOut().getBody() == null) {
                return;
            }

            // Body is not null, is it a String ?
            Object bodyString = exchange.getOut().getBody();
            if (bodyString instanceof String) {
                response.getWriter().write((String) bodyString);
                return;
            }

            // Body is not a String, is it an InputStream ?
            InputStream bodyStream = exchange.getOut().getBody(InputStream.class);
            if (bodyStream != null) {
                try {
                    int read;
                    while ((read = bodyStream.read()) != -1) {
                        response.getOutputStream().write(read);
                    }
                } finally {
                    bodyStream.close();
                    return;
                }
            }

            // Body is not an InputStream, is it a byte array ?
            byte[] bodyByteArray = exchange.getOut().getBody(byte[].class);
            if (bodyByteArray == null) {
                throw new NullPointerException("Request body is not a String, InputStream or byte array: "
                        + exchange.getOut().getBody().getClass());
            }
            response.getOutputStream().write(bodyByteArray);
        } catch (Throwable t) {
            throw new ServletException("CAMEL process failed: " + t.getMessage(), t);
        } finally {
            Thread.currentThread().setContextClassLoader(oldTCCL);
        }
    }

    public String getServletInfo() {
        return this.toString();
    }

    public void destroy() {
        // Nothing to do
    }

    /**
     * Check some headers from the out message.<br/>
     * Only the following are implemented :<br/>
     * <ul>
     * <li>Status code</li>
     * </ul>
     *
     * @param exchangeMessage the camel exchange message to analyse
     * @param response        the HTTP response on which to work
     */
    private void checkHeaders(Exchange exchangeMessage, HttpServletResponse response) {

        // check error status code
        Integer errorCode = exchangeMessage.getOut().getHeader(CamelServlet.ERROR_STATUS_CODE, Integer.class);
        if (null != errorCode) {
            response.setStatus(errorCode.intValue());
        }

    }

}
