/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelContextServiceFactory.java 20667 2011-01-03 17:07:21Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.service.impl;

import java.util.HashMap;
import java.util.Map;

import org.osgi.framework.Bundle;
import org.osgi.framework.ServiceFactory;
import org.osgi.framework.ServiceRegistration;
import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * CamelContext service factory.
 * 
 * @author Guillaume Porcher
 */
public class CamelContextServiceFactory implements ServiceFactory {

    /**
     * Logger.
     */
    private final Log logger = LogFactory.getLog(CamelContextServiceFactory.class);

    /**
     * Camel context service.
     */
    private ICamelService camelService = null;

    /**
     * Map of all the contexts created by the {@link ServiceFactory}.
     */
    private Map<Bundle, String> camelBundleContexts = new HashMap<Bundle, String>();

    /**
     * Get a CamelContext. A new camel context is created for each requiring
     * bundle.
     * 
     * @see org.osgi.framework.ServiceFactory#getService(org.osgi.framework.Bundle,
     *      org.osgi.framework.ServiceRegistration)
     * @param bundle The bundle using the service.
     * @param registration The ServiceRegistration object for the service.
     * @return the camel context created for the bundle
     */
    public synchronized Object getService(final Bundle bundle, final ServiceRegistration registration) {
        if (!this.camelBundleContexts.containsKey(bundle)) {
            try {
                this.camelBundleContexts.put(bundle, this.camelService.startNewContext(bundle.getBundleContext()));
            } catch (Exception e) {
                this.logger.error("Camel context creation failed for bundle {0}.", bundle);
            }
        }
        String contextName = this.camelBundleContexts.get(bundle);
        return this.camelService.getCamelContext(contextName);
    }

    /**
     * Stop the camel context for the bundle.
     * 
     * @see org.osgi.framework.ServiceFactory#ungetService(org.osgi.framework.Bundle,
     *      org.osgi.framework.ServiceRegistration, java.lang.Object)
     * @param bundle The bundle releasing the service.
     * @param registration The ServiceRegistration object for the service.
     * @param camelContext The CamelContext returned by a previous call to
     *        {@link CamelContextServiceFactory#getService(Bundle, ServiceRegistration)}
     *        .
     */
    public void ungetService(final Bundle bundle, final ServiceRegistration registration, final Object camelContext) {
        String contextName = this.camelBundleContexts.get(bundle);
        try {
            this.camelService.stop(contextName);
            this.logger.debug("The camel context {0} has been stopped.", contextName);
        } catch (Exception e) {
            this.logger.warn("Cannot stop the camel context {0}.", contextName, e);
        }
        this.camelBundleContexts.remove(bundle);
    }

    /**
     * @param camelService the camelService to set
     */
    public void setCamelService(final ICamelService camelService) {
        this.camelService = camelService;
    }
}
