/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: FailureResistentExceptionHandlerProcessor.java 20662 2010-12-22 10:12:24Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.component;

import java.util.logging.Logger;

import org.apache.camel.Exchange;
import org.apache.camel.Message;
import org.apache.camel.Processor;
import org.apache.camel.Producer;
import org.apache.camel.util.ExchangeHelper;

/**
 * Camel processor for error handling which is resistent to failures.<br/>
 * <br/>
 * Let's assume a route like the following:
 * <ul>
 * <code>
 * from("file:///a?delete=true").to("file:///b");<br/>
 * onException(Throwable.class).handled(true).to("file:///data/error");
 * </code>
 * </ul>
 * If for some reason the <code>/data/error</code> folder is not accessible, the
 * <code>onException</code> clause will stop; but will have set
 * <code>handled</code> to <code>true</code>. As a result, the file will be
 * <b>LOST</b>.<br/>
 * <br/>
 * In order to avoid such a case, this processor handles such failures and makes
 * sure the file in folder <code>/a</code> stays there after the failure (as a
 * result, stays "not processed yet").
 */
public class FailureResistentExceptionHandlerProcessor implements Processor {

    private Logger logger = Logger.getLogger(this.getClass().getName());

    private final String destination;

    private final String toString;

    public FailureResistentExceptionHandlerProcessor(final String destination) {
        this.destination = destination;

        StringBuilder toString = new StringBuilder();
        toString.append(this.getClass().getName());
        toString.append("[destination=");
        toString.append(destination);
        toString.append("]");
        this.toString = toString.toString();
    }

    public void process(final Exchange exchange) throws Exception {
        if (ExchangeHelper.isFailureHandled(exchange)) {
            this.logger.warning("ExchangeHelper.isFailureHandled(exchange) is true for exchange " + exchange);
            return;
        }

        try {
            Message errorMessage = exchange.getIn().copy();
            Producer errorProducer = exchange.getContext().getEndpoint(this.destination).createProducer();
            Exchange errorExchange = errorProducer.createExchange();
            errorExchange.setIn(errorMessage);
            errorProducer.process(errorExchange);

            ExchangeHelper.setFailureHandled(exchange);
            exchange.setProperty(Exchange.ERRORHANDLER_HANDLED, Boolean.TRUE);
        } catch (Throwable t) {
            Thread.sleep(5000);
            throw new Exception("Error handling failed", t);
        }
    }

    @Override
    public String toString() {
        return this.toString;
    }
}
