/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelServiceTestComponent.java 20667 2011-01-03 17:07:21Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.service.itests;

import java.io.File;

import org.apache.camel.CamelContext;
import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.osgi.framework.BundleContext;
import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class CamelServiceTestComponent {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The OSGi bundle context. Obtained via constructor.
     */
    protected BundleContext bundleContext = null;

    /**
     * The camel service. Injected by the container.
     */
    private ICamelService camelService = null;

    private String camelContextName;

    /**
     * The JONAS_BASE/logs folder.
     */
    private File jonasLogs;

    public CamelServiceTestComponent(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    /**
     * Start the route.
     */
    public void start() throws Throwable {
        String jonasBase = System.getProperty("jonas.base");
        if (jonasBase == null) {
            throw new IllegalStateException("jonas.base is not set!");
        }
        final File jonasBaseFile = new File(jonasBase);
        if (!jonasBaseFile.isDirectory()) {
            throw new IllegalStateException(jonasBaseFile + " is not a directory");
        }

        this.jonasLogs = new File(jonasBaseFile, "logs");
        if (!this.jonasLogs.isDirectory()) {
            throw new IllegalStateException(this.jonasLogs + " is not a directory");
        }

        final File jonasWorkDir = new File(jonasBaseFile, "work");
        if (!jonasWorkDir.isDirectory()) {
            throw new IllegalStateException(jonasWorkDir + " is not a directory");
        }

        // start camel context and get name
        this.camelContextName = this.camelService.startNewContext(this.bundleContext);
        // get camel context from name
        CamelContext camelContext = this.camelService.getCamelContext(this.camelContextName);

        try {
            this.logger.info("Camel context {0} started", this.camelContextName);

            // Route definition
            RouteBuilder builder = new RouteBuilder() {
                @Override
                public void configure() throws Exception {
                    this.from("file://" + jonasWorkDir.getAbsolutePath() + "/inService").process(new Processor() {
                        public void process(final Exchange e) throws Exception {
                            String message = e.getIn().getBody(String.class);
                            CamelServiceTestComponent.this.logger.info("Received File message: {0}", message);
                            File log = new File(CamelServiceTestComponent.this.jonasLogs, message);
                            log.createNewFile();
                        }
                    });
                }
            };

            // Add the route in the camel context.
            camelContext.addRoutes(builder);
        } catch (Throwable t) {
            this.logger.error("Cannot start CamelServiceTestComponent", t);
            throw t;
        }

        // start camel context and get context
        CamelContext camelContext2 = this.camelService.startAndGetNewContext(this.bundleContext);

        try {
            this.logger.info("Camel context {0} started", this.camelContextName);

            // Route definition
            RouteBuilder builder = new RouteBuilder() {
                @Override
                public void configure() throws Exception {
                    this.from("file://" + jonasWorkDir.getAbsolutePath() + "/inService2").process(new Processor() {
                        public void process(final Exchange e) throws Exception {
                            String message = e.getIn().getBody(String.class);
                            CamelServiceTestComponent.this.logger.info("Received File message: {0}", message);
                            File log = new File(CamelServiceTestComponent.this.jonasLogs, message);
                            log.createNewFile();
                        }
                    });
                }
            };

            // Add the route in the camel context.
            camelContext2.addRoutes(builder);
        } catch (Throwable t) {
            this.logger.error("Cannot start CamelServiceTestComponent", t);
            throw t;
        }
    }

    /**
     * Stop the route.
     */
    public void stop() throws Exception {
        this.camelService.stop(this.camelContextName);
    }

}
